*
* SDCX - COCO SDC EXPLORER
* 2016-2018 GUILLAUME MAJOR (guillaume.major@gmail.com)
* BASED ON SIDEKICK BY LUIS ANTONIOSI (retrocanada76@gmail.com)
*

* Version information
VERSION		EQU		'0'
MAJOR		EQU		'4'
MINOR		EQU		'8'

* Compilation flags
			if		!DRAGON
DRAGON		SET		0
			endif
COCO		SET		!DRAGON
FLOPPYDRV	SET		COCO

* Masks
INTMASKS	EQU		%01010000

			if		COCO
* Coco SDC Hardware Addressing - CoCo Scheme
CTRLATCH	EQU		$FF40			; controller latch (write)
CMDREG		EQU		$FF48			; command register (write)
STATREG		EQU		$FF48			; status register (read)
PREG1		EQU		$FF49			; param register 1
PREG2		EQU		$FF4A			; param register 2
PREG3		EQU		$FF4B			; param register 3
DATREGA		EQU		PREG2			; first data register
DATREGB		EQU		PREG3			; second data register
FLSHDAT		EQU		$FF42			; flash data register
FLSHCTRL	EQU		$FF43			; flash control register
			else
* Coco SDC Hardware Addressing - Dragon Scheme
CTRLATCH	EQU		$FF48			; controller latch (write)
CMDREG		EQU		$FF40			; command register (write)
STATREG		EQU		$FF40			; status register (read)
PREG1		EQU		$FF41			; param register 1
PREG2		EQU		$FF42			; param register 2
PREG3		EQU		$FF43			; param register 3
DATREGA		EQU		PREG2			; first data register
DATREGB		EQU		PREG3			; second data register
FLSHDAT		EQU		$FF4A			; flash data register
FLSHCTRL	EQU		$FF4B			; flash control register
			endif

* Other Hardware Addressing
MPIREG		EQU		$FF7F			; multi-pak slot selection
			
* Status Register Masks
BUSY		EQU		%00000001		; set while a command is executing
READY		EQU		%00000010		; set when ready for a data transfer
FAILED		EQU		%10000000		; set on command failure

* Command and Mode Values
			if		COCO
CMDMODE		EQU		$43				; control latch value to enable command mode (Coco)
			else
CMDMODE		EQU		$0B				; control latch value to enable command mode (Dragon)
			endif
CMDREAD		EQU		$80				; read logical sector
CMDWRITE	EQU		$A0				; write logical sector
CMDEX		EQU		$C0				; extended command
CMDEXD		EQU		$E0				; extended command with data block

VIDRAM	 	EQU		$0400			; video RAM
CLRSCH		EQU		80				; clear screen character

KEYUP		EQU		$5E				; UP key
KEYDN		EQU		$0A				; DOWN key
KEYLT		EQU		$08				; LEFT key
KEYRT		EQU		$09				; RIGHT key

KEYSUP		EQU		$5F				; SHIFT-UP keys
KEYSDN		EQU		$5B				; SHIFT-DOWN keys
KEYSRT		EQU		$5D				; SHIFT-RIGHT keys
KEYSLT		EQU		$15				; SHIFT-LEFT keys

KEYESC		EQU		03				; BREAK key
KEYENT		EQU		13				; ENTER key
KEYCLR		EQU		12				; CLEAR key

KEYDRV1		EQU		#'!'+DRAGON
LBLDRV1		EQU		#'1'+DRAGON

DCOPC		EQU		$00EA			; DSKCON operation
DCDRV		EQU		$00EB			; DSKCON drive
DCTRK		EQU		$00EC			; DSKCON track
DCSEC		EQU		$00ED			; DSKCON sector
DCBPT		EQU		$00EE			; DSKCON buffer
DCSTA		EQU		$00F0			; DSKCON status
DCRETRY		EQU		#3				; DSKCON retry count

POLCAT		EQU		$A000			; read keyboard routine
JOYIN		EQU		$A00A			; read joystick routine
POTVAL		EQU		$015A			; joystick pot values
DSKCON		EQU		$C004			; disk access routine

			if		FLOPPYDRV
DRVTBL		EQU		$D89D			; drive access table
ROMSET		EQU		$FFDE			; ROM enabled
ROMCLR		EQU		$FFDF			; ROM disabled

DBUF1		EQU		$0700			; I/O BUFFER #1
NMIFLG		EQU		$0982			; NMI flag
DNMIVC		EQU		$0983			; NMI vector
RDYTMR		EQU		$0985			; motor off timer
DRGRAM		EQU		$0986			; RAM image of DSKREG
DSKREG		EQU		$FF40			; disk control register
FDCREG		EQU		$FF48			; floppy drive status/command register

LD6D4		EQU		$D6D4			; sector formatting data
LD7D1		EQU		$D7D1			; wait for drive ready

* SDC-DOS VARIABLES
SDCDRV0		EQU		$014A
SDCDRV1		EQU		$014B
SDCDRV2		EQU		$014C
SDCDRV3		EQU		$014D

SDCDW0		EQU		$014E
SDCDW1		EQU		$014F
SDCDW2		EQU		$0150
SDCDW3		EQU		$0151
			endif

GETNCH		EQU		$009F
CURPOS		EQU		$0088
CHARAD		EQU		$00A6

SDCDIRMAX	EQU		512				; maximum number of disks per directory

			if		COCO
CRUNCHLN	EQU		$B821
EXECCMD		EQU		$ADC0
EXECCMD2	EQU		EXECCMD+6
DIRTRACK	EQU		17				; directory track
DIRSECT		EQU		3				; first directory sector
DIRSECTMAX	EQU		11				; max directory sector
DIRENTSIZE	EQU		32				; directory entry size
BOOTTRACK	EQU		34				; boot track
BOOTSECT	EQU		1				; boot sector
FLPYSECT	EQU		18				; number of sectors per track (floppy)
FLPYMAX		EQU		3				; maximum floppy drive number
			else
CRUNCHLN	EQU		$8F67
EXECCMD		EQU		$84D6
EXECCMD2	EQU		EXECCMD+6
DIRTRACK	EQU		20				; directory track
DIRSECT		EQU		3				; first directory sector
DIRSECTMAX	EQU		18				; max directory sector
DIRENTSIZE	EQU		25				; directory entry size
BOOTTRACK	EQU		0				; boot track
BOOTSECT	EQU		3				; boot sector
			endif

SCRWID		EQU		$20
SCRHEI		EQU		$10

WINHEI		EQU		(SCRHEI-5)
WINWID		EQU		(SCRWID/2-3)

			ORG		$2800

START		LDU		#USRSTK
			STS		SREG
			
			LDA		WORKDRV			; set DISKCON working drive
			STA		DCDRV

			if		FLOPPYDRV
			JSR		DETECTMPI		; detect the presence of an MPI

			LDA		SDCDRV0			; disable floppy drive
			ANDA	#$7F			; clear bit 7
			STA		SDCDRV0
			endif

			CLR		PANEFOCUS
			CLR		KBUFPOS
			LDD		#1024			; keyboard buffer delay
			STD		KBUFDELAY
			CLR		JOYSEL
			CLR		JOYFLAG
			CLR		JOYTMR
			CLR		JOYBTMR
			CLR		JOYSTA

			LDA		#143+CLRSCH
			JSR		CLRSCR
			
			if		!TEST
			JSR		DETECTSDC
			LBCS	NOSDC
			endif
			
			JSR		LISTSDC			; get SDC directory
			if		!TEST
			JSR		DREAD			; get selected disk directory
			endif

MAINLOOP	JSR		DRAWSCR			; draw screen

READKEY		JSR		[POLCAT]		; read keyboard
			BNE		HANDLEKEY

			LDX		KBUFTMR			; decrement keyboard buffer
			BEQ		READKEY2
			LEAX	-1,X
			STX		KBUFTMR
			BNE		READKEY2

			CLR		KBUFPOS			; clear keyboard buffer
			LDX		#KBUF
			LDD		#0
			STD		,X
			STD		2,X
			STD		4,X

READKEY2	JSR		READJOYBUT		; read joystick button
			BEQ		READKEY3		; exit if no button pressed

			LDA		#KEYENT			; simulate ENTER key
			STA		KEYPRES
			LBRA	GOPANE			; process key

READKEY3	TST		JOYFLAG			; is joystick enabled?
			BEQ		READKEY			; no, exit
			JSR		READJOY			; read joystick axis (converted key in A)
			BEQ		READKEY			; no, exit
			STA		KEYPRES			; save converted key			
			LBRA	NOSHIFTKEY		; handle key (skip SHIFT commands)

HANDLEKEY	STA		KEYPRES

			* TEST THE SHIFT KEY
			LDA		#%01111111		; column 7
			STA		$FF02
			LDA		$FF00
			ANDA	#%01000000		; row 6
			EORA	#%01000000
			STA		SHIFTKEY

			LDA		KEYPRES

			TST		SHIFTKEY		; test SHIFT key
			LBEQ	NOSHIFTKEY		; branch if no SHIFT key
			
			CMPA	#'C'			; SHIFT-C keys
			BEQ		ISCKEY
			
			if		FLOPPYDRV
			CMPA	#'R'			; SHIFT-R keys
			BEQ		ISRKEY
			
			CMPA	#'W'			; SHIFT+W keys
			BEQ		ISWKEY
			
			CMPA	#'D'			; SHIFT-D keys
			BEQ		ISDKEY
			
			CMPA	#'F'			; SHIFT-F keys
			BEQ		ISFKEY			
			endif

			CMPA	#'I'			; SHIFT-I keys
			BEQ		ISIKEY
			
			CMPA	#'K'			; SHIFT-K keys
			BEQ		ISKKEY
			
			CMPA	#'S'			; SHIFT-S keys
			BEQ		ISSKEY
			
			CMPA	#'H'			; SHIFT-H keys
			LBEQ	ISHKEY

			LBRA	GOPANE

ISCKEY		JSR		CREATEDSK		; create disk
			LBRA	MAINEND
	
			if		FLOPPYDRV
ISRKEY		JSR		CHECKDSK		; check if disk can be mounted
			LBCS	MAINEND			; no - exit
			TST		MNTERR			; check if current disk was mounted successfully
			LBNE	MAINEND			; no - exit
			TST		MNTDRV1			; check if current disk is not mounted on drive 1
			LBNE	MAINEND			; yes - exit
			JSR		CHKFLPY			; check if floppy drive operations are possible
			LBCS	MAINEND
			JSR		READFLPY		; read floppy disk
			LBRA	MAINEND

ISWKEY		JSR		CHECKDSK		; check if disk can be mounted
			LBCS	MAINEND			; no - exit
			TST		MNTERR			; check if current disk was mounted successfully
			LBNE	MAINEND			; no - exit
			TST		MNTDRV1			; check if current disk is not mounted on drive 1
			LBNE	MAINEND			; yes - exit
			JSR		CHKFLPY			; check if floppy drive operations are possible
			LBCS	MAINEND
			JSR		WRITEFLPY		; write floppy disk
			LBRA	MAINEND

ISDKEY		JSR		CHKFLPY			; check if floppy drive operations are possible
			LBCS	MAINEND
			JSR		FLOPPYDIR		; list files on floppy disk
			JSR		DREAD			; refresh directory
			BRA		MAINEND

ISFKEY		JSR		FORMATFLPY
			BRA		MAINEND
			endif

ISIKEY		TST		MNTERR			; check if current disk was mounted successfully
			LBNE	MAINEND			; no - exit
			JSR		FILEINFO		; file info
			BRA		MAINEND
			
ISKKEY		JSR		MAKEDIR			; create directory
			BRA		MAINEND

ISSKEY		LDB		DIRSORT			; toggle dirsort value
			EORB	#1
			STB		DIRSORT
			BEQ		MAINEND			; DIRSORT > 0?
			LDB		DIRSIZE			; DIRIZE > 0?
			BEQ		MAINEND
			CLRA					; sort directory
			JSR		DSORT
			BRA		MAINEND

ISHKEY		JSR		HELP
			BRA		MAINEND

NOSHIFTKEY	CMPA	#KEYLT
			BEQ		ISLTKEY
			
			CMPA	#KEYRT
			BEQ		ISRTKEY

			CMPA	#KEYCLR			; CLEAR key
			BEQ		ISCLRKEY
			
			BRA		GOPANE
			
ISLTKEY		TST		PANEFOCUS
			BNE		CHANGEFOCUS
			BRA		MAINEND

ISRTKEY		TST		PANEFOCUS
			BEQ		CHANGEFOCUS
			BRA		MAINEND

ISCLRKEY	TST		MULTIDSK		; refresh disk directory
			BEQ		MAINEND
			JSR		REFRESH
			BRA		MAINEND
			
GOPANE		TST		PANEFOCUS
			BEQ		LEFTPANE
			BRA		RIGHTPANE

CHANGEFOCUS	TST		DIRSIZE
			BEQ		CHANGEF02
			LDA		PANEFOCUS
			COMA
			STA		PANEFOCUS
			CLR		KBUFPOS
CHANGEF02	BRA		MAINEND

LEFTPANE	JSR		DSKLISTKEY
			BRA		ISESCKEY

RIGHTPANE	JSR		DIRLISTKEY

ISESCKEY	LDA		KEYPRES
			CMPA	#KEYESC
			BEQ		EXIT

MAINEND		LBRA	MAINLOOP

**
*  RETURN TO BASIC
**
EXIT		LDX		#CLSCMD			; clear screen command
			JMP		RUNBAS			; execute and return to BASIC

**
*  READ JOYSTICK BUTTONS
*  EXIT: 
*    Z is set if no button is pressed
**
READJOYBUT	CLR		JOYBUT			; reset JOYBUT
			LDA		#$FF			; unselect all keyboard colums
			STA		$FF02			; store value
			LDA		$FF00			; read joystick buttons
			COMA					; flip bits
			ANDA	#$03			; keep only left and right buttons
			BEQ		READJOYB03		; no button detected
			TST		JOYFLAG			; check if joystick is active
			BEQ		ENABLEJOY		; no? enable it

READJOYB02	BITA	JOYSEL			; check if button is selected joystick
			BEQ		READJOYB03		; wrong button, exit

			LDA		JOYSTA			; check joystick button state
			CMPA	#2				; state 2
			BEQ		READJOYB04		; exit
			CMPA	#1				; state 1
			BEQ		READJOYB02B		; handle state 1

			* STATE 0 - button pressed
			LDA		#1
			STA		JOYSTA			; set JOYSTA = 1
			LDA		#100
			STA		JOYBTMR			; start joytick timer
			BRA		READJOYB04		; exit

			* STATE 1 - button pressed
READJOYB02B	DEC		JOYBTMR			; decrement joystick button timer
			BGT		READJOYB04		; exit if joystick button timer > 0
			LDA		#2
			STA		JOYSTA			; set JOYSTA = 2
			BRA		READJOYB04		; exit

READJOYB03	TST		JOYFLAG			; check if joystick is active
			BEQ		READJOYB04		; exit
			LDA		JOYSTA			; check joystick button state
			CMPA	#1				; state 1
			BNE		READJOYB03B

			* STATE 1 - button not pressed
			CLR		JOYSTA
			LDA		#1
			STA		JOYBUT
			RTS

READJOYB03B	CLR		JOYSTA

READJOYB04	CLRA					; return no button pressed (set Z)
			RTS

ENABLEJOY	STA		JOYSEL			; save joystick selection
			LDB		#1				; set joystick flag
			STB		JOYFLAG
			LDD		#256			; shorten keyboard delay to
			STD		KBUFDELAY		; compensate for joystick readings
			LDA		#2				; set state = 2 to cancel button press
			STA		JOYSTA 

			LDD		#$010F			; display joystick enabled message
			LDX		#JOYMSG
			JSR		PRINTRASC
			JSR		FLUSHSCR

			CLRA					; cancel button press (set Z)
			RTS						; exit

**
*  READ JOYSTICK POTS
*  EXIT: 
*    A = converted key press
*    Z is set if selected joystick is centered
**
READJOY		TST		JOYSTA			; check joystick button state
			BNE		READJOY01		; read joystick immediately
			LDA		JOYTMR			; check joystick timer
			BEQ		READJOY01		; read joystick if timer = 0
			DEC		JOYTMR			; decrement joystick timer
			BRA		READJOYEND		; exit

READJOY01	PSHS	U				; save user stack
			JSR		[JOYIN]			; read joystick
			PULS	U				; restore user stack
			LDX		#POTVAL			; load joystick pot values address
			LDA		JOYSEL			; get selected joystick
			CMPA	#1
			BNE		READJOY02
			LDD		,X				; get right joystick coordinates
			BRA		READJOY03

READJOY02	LDD		2,X				; get left joystick coordinates

READJOY03	CMPB	#8				; check UP (Y-Axis)
			BHI		READJOY04
			LDA		#KEYUP
			BRA		READJOYOK

READJOY04	CMPB	#56				; check DOWN (Y-Axis)
			BLO		READJOY05
			LDA		#KEYDN
			BRA		READJOYOK

READJOY05	CMPA	#8				; check LEFT (X-Axis)
			BHI		READJOY06
			LDA		#KEYLT
			BRA		READJOYOK

READJOY06	CMPA	#56				; check RIGHT (X-Axis)
			BLO		READJOYEND
			LDA		#KEYRT

READJOYOK	LDB		#100			; start joystick timer
			STB		JOYTMR
			TSTA					; clear Z
			RTS

READJOYEND	CLRA					; set Z
			RTS

**
*  DRAW THE USER INTERFACE
**
DRAWSCR		JSR		DRAWBG			; draw background

			LDD		#$0102			; draw SDC directory window
			LDX		#WINWID+2
			LDY		#WINHEI+2
			JSR		WINDOW

			LDX		#DRVLABEL		; display SDC directory title
			LDD		#$0102
			JSR		PRINTRASC

			LDD		#$1102			; draw disk directroy window
			LDX		#WINWID+2
			LDY		#WINHEI+2
			JSR		WINDOW

			LDX		#DIRLABEL		; display file directory title
			LDD		#$1102
			JSR		PRINTRASC

			JSR		CLRSTATUS		; clear status bar
			
			LDX		#EXITCMD		; display exit command
			LDD		#$150F
			JSR		PRINTRASC
			
			JSR		DSKLIST			; display SDC directory list
			JSR		DIRLIST			; display disk directory list

			* IF DISK IS BOOTABLE AND NOT AN OS-9 DISK AND FOCUS IS ON
			* THE DISK BOX, DISPLAY A "PRESS ENTER TO BOOT" STATUS MESSAGE
			
			TST		BOOTABLE		; check if disk is bootable
			BEQ		DRAWSCR02		; no - exit
			TST		PANEFOCUS		; check is focus is on the disk box
			BNE		DRAWSCR02		; no - exit
			TST		OS9DSK			; check if disk is an OS-9 disk
			BNE		DRAWSCR02		; yes - exit
			LDX		#BOOTCMD		; display boot command
			BRA 	DRAWSCR03

DRAWSCR02	LDX		#HELPCMD		; display help command

DRAWSCR03	LDD		#$010F			; update status bar
			JSR		PRINTRASC

			TST		DIRSORT			; check if disk directory is sorted
			BEQ		DRAWSCR04		; no - exit
			LDX		#SORTEDLBL		; display up arrow on the window title
			LDD		#$1D02
			JSR		PRINTRASC

DRAWSCR04	JSR		FLUSHSCR		; update screen

			LDA		JOYBTMR			; decrement joystick button timer
			SUBA	#10				; to compensate for lost cycles
			BHS		DRAWSCR05		; for drawing the screen
			CLRA

DRAWSCR05	STA		JOYBTMR
			RTS

**
*  DRAW UI BACKGROUND
**
DRAWBG		LDA		#143+CLRSCH
			JSR		CLRSCR

			LDX		#PRGTITLE
			CLRA
			CLRB
			JSR		PRINTRASC
			
			RTS
			
**
*  COPY SCREEN BUFFER TO VIDEO MEMORY
**
FLUSHSCR	PSHU	X,Y,A,B,CC
			LDD		#32*16
			LDX		#SCRBUF
			LDY		#VIDRAM
			JSR		MEMCPY
			PULU	X,Y,A,B,CC
			RTS

**
*  DISPLAY SDC DIRECTORY
**
DSKLIST		LDD		DSKSIZE
			LBEQ	DSKLISTEMP
			SUBD	DSKOFF
			CMPD	#WINHEI
			BLO		DSKLIST01
			LDD		#WINHEI

DSKLIST01	LDY		#0
			LEAY	D,Y

			LDB		#3
			STB		CURLINE			; start line

			LDD		DSKOFF
			JSR		GETDSKADR

DSKLIST02	PSHU	X,Y,A,B
			LDY		#STR1
			LDA		#' '
			STA		,Y+

			* FILENAME
			LDD		#8
			JSR		MEMCPY
			LEAX	8,X
			LEAY	8,Y

			LDA		3,X				; check type
			BITA	#$10
			BNE		DSKLIST03

			* DOT
			LDA		#'.'
			STA		,Y+

			* EXTENSION
			LDD		#3
			JSR		MEMCPY
			LEAY	3,y
			BRA		DSKLIST04

DSKLIST03
			* DIR
			LDA		#' '
			STA		,Y+
			LDA		#'<'
			STA		,Y+
			LDA		#'D'
			STA		,Y+
			LDA		#'>'
			STA		,Y+

			* SPACER
DSKLIST04	LDA		#' '
			STA		,Y+

			* NULL TERMINATOR
			CLRA
			STA		,Y+

			CLRA
			LDB		CURLINE
			SUBB	#3
			ADDD	DSKOFF
			CMPD	DSKSEL
			BEQ		DSKLIST05

			* PRINT LINE
			LDB		CURLINE			; curr line
			LDA		#1
			LDX		#STR1
			JSR		PRINTASC
			PULU	X,Y,A,B
			BRA		DSKLIST06

			* PRINT REVERSE LINE
DSKLIST05	LDB		CURLINE			; curr line
			LDA		#1
			LDX		#STR1
			JSR		PRINTRASC
			PULU	X,Y,A,B

DSKLIST06	LEAX	16,X
			INC		CURLINE
			LEAY	-1,Y
			BNE		DSKLIST02

DSKLISTEND	RTS

DSKLISTEMP	LDB		#4
			LDA		#2
			LDX		#EMPTYCARD
			JSR		PRINTASC
			
			RTS

**
*  DISPLAY SELECTED DISK DIRECTORY
**
DIRLIST		TST		MNTERR
			BNE		DIRMNTERR

			TST		DSKERR
			BNE		DIRMNTERR

			TST		MNTDRV1
			BNE		DIRMNTDRV1

			TST		OS9DSK
			BNE		DIROS9DSK

			LDB		DIRSIZE
			BEQ		DIRLISTEND
			SUBB	DIROFF
			CMPB	#WINHEI
			BLO		DIRLIST01
			LDB		#WINHEI

DIRLIST01	LDY		#0
			LEAY	B,Y

			LDB		#3
			STB		CURLINE			; start line

			LDB		DIROFF
			JSR		GETDIRADR

DIRLIST02	PSHU	X,Y

			TFR		X,Y
			LDX		#STR1			; get filename
			JSR		FORMATDFN

			LDA		#SCRWID/2+1
			TST		PANEFOCUS		; check if disk directory list has focus
			BEQ		DIRLIST03
			
			LDB		CURLINE
			SUBB	#3
			ADDB	DIROFF
			CMPB	DIRSEL
			BEQ		DIRLIST04

DIRLIST03	LDB		CURLINE			; load current line
			JSR		PRINTASC		; print standard text
			BRA		DIRLIST05

DIRLIST04	LDB		CURLINE			; load current line
			JSR		PRINTRASC		; print reversed text

DIRLIST05	PULU	X,Y

			LEAX	16,X			; next file
			INC		CURLINE
			LEAY	-1,Y
			BNE		DIRLIST02

DIRLISTEND	RTS

DIRMNTERR	LDB		#4
			LDA		#SCRWID/2+2
			LDX		#MNTERRMSG
			JSR		PRINTASC

			RTS

DIRMNTDRV1	LDB		#4
			LDA		#SCRWID/2+2
			LDX		#MNTDRV1MSG
			JSR		PRINTASC

			LDB		#5
			LDA		#SCRWID/2+2
			LDX		#MNTDRV1MSG2
			JSR		PRINTASC

			RTS

DIROS9DSK	LDB		#4
			LDA		#SCRWID/2+3
			LDX		#OS9DSKMSG
			JSR		PRINTASC

			TST		BOOTABLE
			BEQ		DIROS9END

			LDB		#6
			LDA		#SCRWID/2+2
			LDX		#OS9BOOTMSG
			JSR		PRINTASC

			LDB		#7
			LDA		#SCRWID/2+4
			LDX		#OS9BOOTMSG2
			JSR		PRINTASC

DIROS9END	RTS

**
*  HANDLE SDC DIRECTORY KEY
**
DSKLISTKEY	LDA		KEYPRES

			CMPA	#KEYUP			; UP key
			BEQ		ISUPKEY
			
			CMPA	#KEYDN			; DOWN key
			BEQ		ISDNKEY
			
			CMPA	#KEYSDN			; SHIFT-DOWN keys
			LBEQ	ISSDNKEY

			CMPA	#KEYSUP			; SHIFT-UP keys
			LBEQ	ISSUPKEY						

			CMPA	#KEYSLT			; SHIFT-LEFT keys
			LBEQ	ISSLTKEY

			CMPA	#KEYSRT			; SHIFT-RIGHT keys
			LBEQ	ISSRTKEY			

			CMPA	#KEYENT			; ENTER key
			LBEQ	ISENTKEY
			
			TST		SHIFTKEY		; SHIFT key
			LBEQ	ISALPHA
			
			CMPA	#'X'			; X key
			LBEQ	ISXKEY

			CMPA	#'N'			; N key
			LBEQ	ISNKEY			
			
			CMPA	#'M'			; M key
			LBEQ	ISMKEY

			CMPA	#KEYDRV1		; SHIFT-1 (Coco), SHIFT-2 (Dragon)
			LBEQ	IS1KEY			
			
			LBRA	READEND			

ISUPKEY		LDD		DSKSEL
			LBEQ	READEND
			SUBD	#1
			STD		DSKSEL

			LDD		DSKOFF
			CMPD	DSKSEL
			LBLS	DREAD

			SUBD	#1
			STD		DSKOFF

			JMP		DREAD

ISDNKEY		LDD		DSKSEL
			ADDD	#1
			CMPD	DSKSIZE
			LBEQ	READEND

			STD		DSKSEL

			SUBD	DSKOFF
			CMPD	#WINHEI
			LBLO	DREAD
			LDD		DSKOFF
			ADDD	#1
			STD		DSKOFF

			JMP		DREAD

ISSDNKEY	LDD		DSKSIZE
			SUBD	DSKSEL
			CMPD	#1
			LBEQ	READEND
			CMPD	#WINHEI
			BCC		ISSDNKEY02

			LDD		DSKSIZE
			SUBD	#1
			BRA		ISSDNKEY03

ISSDNKEY02	LDD		DSKSEL
			ADDD	#WINHEI-1

ISSDNKEY03	STD		DSKSEL
			JSR		SETDSKOFF
			JMP		DREAD

ISSUPKEY	LDD		DSKSEL
			CMPD	#WINHEI-1
			BCC		ISSUPKEY02
			LDD		#WINHEI-1

ISSUPKEY02	SUBD	#WINHEI-1
			STD		DSKSEL
			STD		DSKOFF

			JMP		DREAD

ISSLTKEY	LDD		#0
			STD		DSKSEL
			STD		DSKOFF

			JMP		DREAD

ISSRTKEY	LDD		DSKSIZE
			SUBD	#1
			STD		DSKSEL
			JSR		SETDSKOFF

			JMP		DREAD

ISENTKEY	TST		BOOTABLE		; check if disk is bootable
			LBNE	BOOTDSK			; yes - boot disk
			JSR		CHDIR			; change directory
			JMP		READEND

ISXKEY		JSR		DELDSK			; delete disk
			JMP		READEND

ISNKEY		JSR		RENAME			; rename disk
			JMP		READEND

ISMKEY		JSR		MOUNTDIR		; mount directory
			JMP		READEND

IS1KEY		JSR		CHECKDSK		; check if disk can be mounted
			BCS		IS1KEYEND		; no - exit
			LDA		MNTDRV1			; load disk mounted in drive 1 flag
			STA		SDCDRV			; set SDC drive
			JSR		UNMOUNT			; unmount disk
			LBCS	SDCERR			; branch on error
			CLR		SDCDRV			; set SDC drive to 0
			TST		MNTDRV1			; check disk mounted in drive 1 flag
			LBNE	DREAD			; mount disk in drive 0 and list dir
			LDA		#1				; set SDC drive to 1
			STA		SDCDRV
			JSR		MOUNTDSK		; mount disk in drive 1
			BCS		IS1KEYEND		; branch on error
			LDA		#1				; set disk mounted in drive 0 flag
			STA		MNTDRV1
			
IS1KEYEND	CLR		SDCDRV			; set SDC drive to 0
			JMP		READEND

ISALPHA		CMPA	#'A'
			BLO		ISNUM
			CMPA	#'Z'
			LBHI	READEND
			BRA		ISALPHA01

ISNUM		CMPA	#'0'
			LBLO	READEND
			CMPA	#'9'
			LBHI	READEND

ISALPHA01	LDB		KBUFPOS
			CMPB	#4				; max 4 chars
			LBEQ	READEND

			LDX		#KBUF
			CMPA	,X				; check if key is same than first
			BNE		ISALPHA02
			CMPB	#1				; check if only one key in buffer
			BEQ		ISALPHA03		; don't add key to buffer

ISALPHA02	LEAX	B,X				; store key in buffer
			STA		,X
			INCB
			STB		KBUFPOS

ISALPHA03	LDX		DSKSEL
			CMPB	#1
			BNE		ISALPHA04

			LEAX 	1,X
			CMPX	DSKSIZE			; end of list?
			BNE		ISALPHA04
			LDX		#0

ISALPHA04	STX		KEYSRCH

			LDD		KBUFDELAY		; keyboard delay
			STD		KBUFTMR
			CLR		SRCHPASS

			LDD		KEYSRCH			; get disk address
			JSR		GETDSKADR		; disk adr in x
			LDY		#KBUF
			
ISALPHA05	LDB		KBUFPOS			
			JSR		STRNCMP			; compare buffer with disk entry
			BEQ		ISALPHA07		; match?

			LDD		KEYSRCH			; next entry
			ADDD	#1
			CMPD	DSKSIZE
			BEQ		ISALPHA06		; end of list?

			STD		KEYSRCH
			LEAX	16,X
			BRA		ISALPHA05
			
ISALPHA06	TST		SRCHPASS
			BNE		READEND			; second pass, we're done

			LDA		#1				; start from beginning
			STA		SRCHPASS
			LDD		#0
			STD		KEYSRCH
			LDX		PDISKARR
			BRA		ISALPHA05

ISALPHA07	LDD		KEYSRCH
			STD		DSKSEL
			JSR		SETDSKOFF

DREAD		CLR		MULTIDSK
			CLR		MNTDRV1
			CLR		DIRSIZE
			CLR		OS9DSK
			CLR		BOOTABLE
			CLR		DSKERR
			CLR		MNTERR

			LDD		DSKSIZE
			BEQ		READEND
			
			JSR		CHECKDSK
			BCS		READEND
			if		!TEST
			JSR		MOUNTDSK
			BCS		READEND			; branch on error
			endif
			JSR		READDIR

			* SORT DIRECTORY
			TST		DIRSORT
			BEQ		READEND
			LDB		DIRSIZE			; DIRSIZE > 0?
			BEQ		READEND
			CLRA
			JSR		DSORT
			
READEND		LDX		DSKSEL			; check if disk selection changed
			CMPX	DSKSELPREV
			BEQ		READEND2
			LDB		#2
			STB		JOYSTA
			
READEND2	STX		DSKSELPREV		; save previous disk selected
			RTS

**
*  GET DISK ADDRESS
*    D = DISK NUMBER
*  EXIT:
*    X = DISK ADDRESS
**		
GETDSKADR	LSLB					; mul by 16
			ROLA
			LSLB
			ROLA
			LSLB
			ROLA
			LSLB
			ROLA

			LDX		PDISKARR
			LEAX	D,X

			RTS

**
*  GET SELECTED DISK TYPE
*  EXIT: X - TYPE STRING
**
GETDSKTYPE	LDD		DSKSEL			; get selected disk
			JSR		GETDSKADR		; get selected disk address
			LDA		11,X			; check file type
			BITA	#$10			; directory bit
			BNE		GETDSKTYPE2
			LDX		#FILELBL		; load file label
			RTS

GETDSKTYPE2 LDX		#DIRLBL			; load directory label
			RTS

**
*  GET SELECTED DISK SIZE
*  EXIT: X - FORMATTED SIZE WITH UNIT
**
GETDSKSIZE	LDD		DSKSEL			; get selected disk
			JSR		GETDSKADR		; get selected disk address
			
			LDA		11,X			; check file type
			BITA	#$10			; directory bit
			BNE		GETDSKSI04		; branch if it's a directory

			JSR		GETDSKINFO		; get disk information
			
			LDA		31,X			; load disk size (1st MSB)
			LDB		30,X			; load disk size (2nd MSB)
			CMPD	#$009C			; 10 MB
			BLO		GETDSKSI02		; branch if lower
		
			LSRA
			RORB					; shift D 4 bits right
			LSRA					; (divide by 16)
			RORB
			LSRA
			RORB
			LSRA
			RORB
			ADCB	#0				; add carry to round value
			
			LDX		#MBLBL			; load mega bytes label
			PSHU	X				; save on stack
			
			BRA		GETDSKSI03
			
GETDSKSI02	LDA		30,X			; load disk size (2nd MSB)
			LDB		29,X			; load disk size (3rd MSB)
			LSRA
			RORB					; shift D 2 bits right
			LSRA					; (divide by 4)
			RORB
			ADCB	#0				; add carry to round value
			
			LDX		#KBLBL			; load kilo bytes label
			PSHU	X				; save on stack			
			
GETDSKSI03	LDX		#STR1			; load string buffer
			JSR		ITOA16			; convert to ASCII
			JSR		TRIMZERO		; trim leading zeros
			TFR		X,Y				; transfer buffer in Y
			PULU	X				; restore unit label
			JSR		STRCAT			; append to buffer
			TFR		Y,X				; transfer buffer back to X
			RTS

GETDSKSI04	LDX		#NALBL
			RTS

**
*  GET SELECTED DISK INFO
*  EXIT: 
*    Disk information is returned in SECBUF
*    Carry set on error
*    X=disk information data (SECBUF)
**
GETDSKINFO	LDD		DSKSEL			; get selected disk
			JSR		GETDSKADR		; get selected disk address
			
			if		!TEST
			LDA		#CMDEX			; extended command
			LDB		#'I'			; set parameter #1
			LDX		#SECBUF			; load data buffer
			JSR		SENDCMD			; send command to the SDC
			BCS		GETDSKSI02		; branch on error
			else
			LDD		#$0076			; load fake value for testing
			STD		28,X
			LDD		#$0200
			STD		30,X
			endif

			CLRA					; clear carry
			RTS

GETDSKI02	ORCC	#1				; set carry
			RTS

**
*  GET DISK INDEX OF FILENAME IN DSKFN
*  EXIT:
*    D=DISK INDEX (-1 if disk was not found)
**
DSKINDEX	LDD		#0
			STD		KEYSRCH

DSKINDEX01	JSR		GETDSKADR		; get disk address
			LDY		#DSKFN			; mounted disk file name
			LDB		#11				; compare 11 characters
			JSR		STRNCMP			; compare strings
			BNE		DSKINDEX02		; match not found?
			LDD		KEYSRCH			; return index
			RTS
		
DSKINDEX02	LDD		KEYSRCH			; increment disk number
			ADDD	#1
			CMPD	DSKSIZE			; end reached?
			BEQ		DSKINDEX03
			STD		KEYSRCH
			BRA		DSKINDEX01
			
DSKINDEX03	LDD		#-1				; not found
			RTS

**
*  EXPAND FILENAME
*    X = SOURCE STRING
**
EXPFN		LDY		#DSKFN			; fill file name with spaces
			LDA		#' '
			LDB		#11

EXPFN01		STA		,Y+
			DECB
			BNE		EXPFN01
			
			LDY		#DSKFN
			
			LDB		#11
			
EXPFN02		LDA		,X+				; load source char
			BEQ		EXPFN04			; exit if end of string
			CMPA	#'.'			; check for dot
			BNE		EXPFN03			; branch if no dot
			
			LDY		#DSKFN+8		; jump to extention
			LDB		#3				; extension is 3 char
			BRA		EXPFN02
			
EXPFN03		STA		,Y+
			DECB
			BNE		EXPFN02
			
EXPFN04		RTS			


**
*  FORMAT FILENAME
*    X = DESTINATION STRING
*    Y = SOURCE STRING
**
FORMATFN	PSHU	X,Y
			LDB		#8				; copy filename
			
FORMAT02	LDA		,Y+				; load source char
			STA		,X+				; save char to destination
			DECB					; decrement character count
			BNE		FORMAT02		; branch if not done
			
			CLR		,X				; terminate string
			LEAX	-8,X			; point to begin of string
			
			JSR		TRIM			; trim spaces

			LDA		,Y
			CMPA	#' '			; is it a space?
			BEQ		FORMAT05		; yes - exit
			
FORMAT03	TST		,X+				; find end of string
			BNE		FORMAT03

			LEAX	-1,X
			LDA		#'.'
			STA		,X+
			
			LDB		#3				; copy extension
			
FORMAT04	LDA		,Y+				; load source char
			STA		,X+				; save char to destination
			DECB					; decrement character count
			BNE		FORMAT04		; branch if not done
			
			CLR		,X				; terminate string
			
FORMAT05	PULU	X,Y
			RTS			

			
**
*  SET DISK OFFSET
**
SETDSKOFF	LDD		DSKSIZE
			SUBD	DSKSEL
			CMPD	#WINHEI
			BCC		SETDSK02

SETDSK01	LDD		DSKSIZE
			SUBD	#WINHEI
			BCS		SETDSK03

			STD		DSKOFF
			RTS

SETDSK02	LDD		DSKSEL
			CMPD	#WINHEI
			STD		DSKOFF
			RTS

SETDSK03	LDD		#0
			STD		DSKOFF
			RTS

**
*  HANDLE DISK DIRECTORY LIST KEY
**
DIRLISTKEY	LDA		KEYPRES

			CMPA	#KEYUP			; UP key
			BEQ		#DISUPKEY
			
			CMPA	#KEYDN			; DOWN key
			BEQ		DISDNKEY
			
			CMPA	#KEYSDN			; SHIFT-DOWN keys
			BEQ		DISSDNKEY
			
			CMPA	#KEYSUP			; SHIFT-UP keys
			LBEQ	DISSUPKEY
			
			CMPA	#KEYSLT			; SHIFT-LEFT keys
			LBEQ	DISSLTKEY			
			
			CMPA	#KEYSRT			; SHIFT-RIGHT keys
			LBEQ	DISSRTKEY			

			CMPA	#KEYENT
			LBEQ	DISENTKEY		; ENTER key
			
			CMPA	#'A'			; A-Z keys
			BLO		DISNUM
			CMPA	#'Z'
			LBHI	DREADEND
			
			LBRA	DISALPHA

DISNUM		CMPA	#'0'			; 0-9 keys
			LBLO	DREADEND
			CMPA	#'9'
			LBHI	DREADEND
			
			LBRA	DISALPHA			

DISUPKEY	LDB		DIRSEL
			LBEQ	DREADEND
			DECB
			STB		DIRSEL

			LDA		DIROFF
			CMPA	DIRSEL
			LBLS	DREADEND

			DECA
			STA		DIROFF

			JMP		DREADEND

DISDNKEY	LDB		DIRSEL
			INCB	
			CMPB	DIRSIZE
			LBEQ	DREADEND

			STB		DIRSEL

			SUBB	DIROFF
			CMPB	#WINHEI
			LBLO	DREADEND
			LDA		DIROFF
			INCA
			STA		DIROFF

			JMP		DREADEND

DISSDNKEY	LDB		DIRSIZE
			SUBB	DIRSEL
			CMPB	#WINHEI
			BCC		DISSDBKEY2

			LDB		DIRSIZE
			DECB
			BRA		DISSDBKEY3

DISSDBKEY2	LDB		DIRSEL
			ADDB	#WINHEI-1

DISSDBKEY3	STB		DIRSEL
			JSR		SETDIROFF
			JMP		READEND	

DISSUPKEY	LDB		DIRSEL
			CMPB	#WINHEI-1
			BCC		DISSUPKEY2
			LDB		#WINHEI-1

DISSUPKEY2	SUBB	#WINHEI-1
			STB		DIRSEL
			STB		DIROFF
			JMP		DREADEND

DISSLTKEY	CLR		DIRSEL
			CLR		DIROFF

			JMP		DREADEND

DISSRTKEY	LDB		DIRSIZE
			DECB
			STB		DIRSEL
			JSR		SETDIROFF

			JMP		DREADEND

DISALPHA	LDB		KBUFPOS
			CMPB	#4				; max 4 chars
			LBEQ	DREADEND

			LDX		#KBUF			
			CMPA	,X				; check if key is same than first
			BNE		DISALPHA02
			CMPB	#1				; check if only one key in buffer
			BEQ		DISALPHA03		; don't add key to buffer

DISALPHA02	STA		B,X
			INCB
			STB		KBUFPOS

DISALPHA03	LDA		DIRSEL
			CMPB	#1
			BNE		DISALPHA04

			INCA
			CMPA	DIRSIZE			; end of list?
			BNE		DISALPHA04
			CLRA

DISALPHA04	STA		KEYSRCH

			LDD		KBUFDELAY		; keyboard delay
			STD		KBUFTMR
			CLR		SRCHPASS
			
			LDB		KEYSRCH			; get dir entry address
			JSR		GETDIRADR		; dir entry adr in x
			LDY		#KBUF

DISALPHA05	LDB		KBUFPOS
			JSR		STRNCMP			; compare buffer with dir entry
			BEQ		DISALPHA07		; match?

			LDA		KEYSRCH			; next entry
			INCA
			CMPA	DIRSIZE
			BEQ		DISALPHA06		; end of list?

			STA		KEYSRCH
			LEAX	16,X
			BRA		DISALPHA05
			
DISALPHA06	TST		SRCHPASS
			BNE		DREADEND		; second pass, we're done

			LDA		#1				; start from beginning
			STA		SRCHPASS
			CLR		KEYSRCH
			LDX		#DIRARR
			BRA		DISALPHA05

DISALPHA07	LDA		KEYSRCH
			STA		DIRSEL
			JSR		SETDIROFF
			BRA		DREADEND
			
DISENTKEY	
			if		COCO

			LDB		DIRSEL			; load selected file
			JSR		GETDIRADR		; get selected file address
			
			LDA		13,X			; check file type
			BNE		ISLM			; branch if not BASIC

			PSHU	X				; save selected file address
			
			LDY		#STR1			; load string buffer
			CLR		,Y				; terminate string
			
			LDX		#RUNCMD			; load RUN command
			JSR		STRCAT			; append command text
			
			PULU	X				; restore selected file address
			JSR		STRCAT			; append filename

			LDX		#RUNCMD2		; load RUN command #2
			JSR		STRCAT			; append command text
			
			BRA		RUN07			; go execute command

ISLM		CMPA	#2				; machine language?
			BNE		DREADEND		; no - exit

			PSHU	X				; save selected file address
			
			LDY		#STR1			; load string buffer
			CLR		,Y				; terminate string
			
			LDX		#RUNMCMD		; load LOADM/EXEC command
			JSR		STRCAT			; append command text
			
			PULU	X				; restore selected file address
			JSR		STRCAT			; append filename
			
			LDX		#RUNMCMD2		; load LOADM/EXEC command #2
			JSR		STRCAT			; append command text
			
			else
			
			LDY		#STR1			; load string buffer
			LDX		#RUNCMD			; load RUN command
RUN02		LDA		,X+				; load character from command text
			BEQ		RUN03			; branch on null character
			STA		,Y+				; save character to string buffer
			BRA		RUN02			; copy next character
			
RUN03		LDB		DIRSEL			; load selected file
			JSR		GETDIRADR		; get selected file address
			LDB		#8				; copy 8 characters

RUN04		LDA		,X+				; load filename character
			CMPA	#' '			; check for space
			BEQ		RUN05			; don't copy if it's a space
			STA		,Y+				; save character to string buffer
RUN05		DECB					; decrement character count
			BNE		RUN04			; copy next character

			LDX		#RUNCMD2		; load RUN command #2
RUN06		LDA		,X+				; load character from command text
			STA		,Y+				; save character to string buffer
			BNE		RUN06			; copy next character

			endif

RUN07		LDX		#STR1			; load BASIC command
			JMP		RUNBAS			; execute command

DREADEND	LDA		DIRSEL			; check if file selection changed
			CMPA	DIRSELPREV
			BEQ		DREADEND2
			LDB		#2				; yes. Ignore joystick button
			STB		JOYSTA
			
DREADEND2	STA		DIRSELPREV		; save previous file selected
			RTS

**
*  EXECUTE BASIC COMMAND AND EXIT TO BASIC
*    X=COMMAND
**
RUNBAS		STX		CHARAD			; set BASIC current execution address
			JSR		CRUNCHLN		; crunch BASIC line
			LDS		SREG			; restore system stack
			JMP		EXECCMD			; execute BASIC command

			if		0				; not in use for the moment
**
*  EXECUTE BASIC COMMAND
*    X=COMMAND
**
RUNBAS2		PSHS	U,X,CC			; save registers
			STX		CHARAD			; set BASIC current execution address
			JSR		CRUNCHLN		; crunch BASIC line
RUNBAS202	JSR		GETNCH			; get next BASIC character
			JSR		EXECCMD2		; execute command
			BNE		RUNBAS202		; loop until done
			PULS	U,X,CC			; restore registers
			RTS
			endif
			
**
*  SET DIRECTORY OFFSET
**		
SETDIROFF	LDA		DIRSIZE
			SUBA	DIRSEL
			CMPA	#WINHEI
			BCC		SETDIR02

			LDA		DIRSIZE
			SUBA	#WINHEI
			BCS		SETDIR03

			STA		DIROFF
			RTS

SETDIR02	LDA		DIRSEL
			STA		DIROFF
			RTS

SETDIR03	CLRA
			STA		DIROFF
			RTS


			if		COCO
**
*  READ SELECT DISK DIRECTORY (COCO)
**
READDIR		CLR		DIRSIZE
			CLR		DIRSEL
			CLR		DIROFF
			CLR		OS9DSK
			CLR		BOOTABLE
			CLR		DSKERR
			
			* CHECK IF SELECTED DISK IS SDF
			
			if		FLOPPYDSK
			TST		FLOPPY			; floppy drive operation?
			BNE		READDIR02		; yes - skip
			endif

			JSR		GETDSKINFO		; get disk information
			LDA		11,X			; get file attributes
			BITA	#$04			; SDF bit?
			BNE		READDIR04		; yes - skip OS-9 check
			
			* READ LSN0
				
READDIR02	LDD		#$0001			; track 0 sector 1
			JSR		READSECT		; read sector
			BCC		READDIR03		; branch if no error
			BRA		READDIR04		; skip OS-9 check

			* CHECK IF OS-9 DISK
			
READDIR03	LDD		,X				; number of sectors (DD.TOT)
			CMPD	#$0800			; 128MB
			BHI		READDIR04
			LDA		3,X				; track size in sectors (DD.TKS)
			CMPA	#$12			; 18 sectors
			BNE		READDIR04
			LDD		$11,X			; number of sectors per track (DD.SPT)
			CMPD	#$0012			; 18 sectors
			BNE		READDIR04

			* IT IS AN OS-9 DISK
			LDA		#1
			STA		OS9DSK
			BRA		READDIR07		; skip RS-DOS file listing
			
READDIR04	LDY		#DIRARR
			JSR		FINDFIRST
			BCS		READDIR07

			LDA		,X				; load first character
			BEQ		READDIR06		; go find next file if deleted entry

			CMPA	#$FF			; check of end of directory
			BEQ		READDIR07		; yes - exit

READDIR05	LDD		#8				; copy filename
			JSR		MEMCPY
			LEAY	D,Y
			LEAX	D,X
			LDB		#'.'			; insert dot
			STB		,Y+
			LDB		#3				; copy extension
			JSR		MEMCPY
			LEAY	3,Y
			CLRB
			STB		,Y+

			LDA		3,X				; get file type
			STA		,Y

			LEAY	(16-13),Y		; point to next file
			INC		DIRSIZE			; increment file count

READDIR06	JSR		FINDNEXT
			BCS		READDIR07
			LDA		,X
			BEQ		READDIR07
			CMPA	#$FF
			BNE		READDIR05

			* READ BOOT SECTOR
			
READDIR07	LDA		#BOOTTRACK		; boot track number
			LDB		#BOOTSECT		; boot sector number
			JSR		READSECT		; read sector
			BCS		RDIREND			; branch on error

			* CHECK IF DISK IS BOOTABLE
			LDD		,X
			CMPD	#$4F53			; OS
			BNE		RDIREND

			* DISK IS BOOTABLE
			LDA		#1
			STA		BOOTABLE

RDIREND		RTS

RDIRERR		LDA		#1
			STA		DSKERR
			RTS

			else

**
*  READ SELECTED DISK DIRECTORY (DRAGON)
**
READDIR		PSHU	X,Y,A,B,CC

			CLR		DIRSIZE
			CLR		DIRSEL
			CLR		DIROFF
			CLR		OS9DSK
			CLR		BOOTABLE
			CLR		DSKERR

			* READ BOOT SECTOR
			LDA		#BOOTTRACK
			LDB		#BOOTSECT
			JSR		READSECT		; disk buffer address returned in x
			BCS		ISOS9DSK		; skip boot check on error

			* CHECK IF DISK IS BOOTABLE
			LDD		,X
			CMPD	#$4F53			; OS
			BNE		ISOS9DSK

			* DISK IS BOOTABLE
			LDA		#1
			STA		BOOTABLE

			* CHECK IF OS-9 DISK

ISOS9DSK	LDD		#$0001			; track 0 sector 1
			JSR		READSECT		; disk buffer address returned in x
			BCS		READDIR02		; skip OS-9 detection on error

			LDD		,X				; number of sectors (DD.TOT)
			CMPD	#$0800			; 128MB
			BHI		READDIR02
			
			LDA		3,X				; track size in sectors (DD.TKS)
			CMPA	#$12			; 18 sectors
			BNE		READDIR02
			
			LDD		$11,X			; number of sectors per track (DD.SPT)
			CMPD	#$0012			; 18 sectors
			BNE		READDIR02

			* IT IS AN OS-9 DISK
			LDA		#1
			STA		OS9DSK
			BRA		RDIREND

READDIR02	LDY		#DIRARR
			JSR		FINDFIRST
			BCS		RDIREND

			LDA		,X
			BITA	#$08			; check for end directory bit
			BNE		RDIREND
			BITA	#$81			; check for continuation and deleted entry bits
			BNE		READDIR08

READDIR03	LEAX	1,X
			LDB		#8

READDIR04	LDA		,X+
			BNE		READDIR05
			LDA		#' '			; replace 0x00 by space

READDIR05	STA		,Y+
			DECB
			BNE		READDIR04

			LDA		#'.'
			STA		,Y+

			LDB		#3

READDIR06	LDA		,X+
			BNE		READDIR07
			LDA		#' '			; replace 0x00 by space

READDIR07	STA		,Y+
			DECB
			BNE		READDIR06

			CLRA
			STA		,Y+

			LEAY	(16-13),Y

			LDA		DIRSIZE
			INCA
			STA		DIRSIZE

READDIR08	JSR		FINDNEXT
			BCS		RDIREND

			LDA		,X
			BITA	#$08			; check for end directory bit
			BNE		RDIREND
			BITA	#$81			; check for continuation and deleted entry bits
			BNE		READDIR08
			BRA		READDIR03

RDIREND		PULU	X,Y,A,B,CC
			RTS
			endif

**
*  FIND FIRST DIRECTORY ENTRY
**
FINDFIRST	PSHU	Y,A,B

			CLR		SECPOS
			LDA		#DIRTRACK
			LDB		#DIRSECT
			STB		CURSEC
			JSR		READSECT
			BCC		FINDFI02		; branch if no error
			LDA		#1
			STA		DSKERR

FINDFI02	PULU	Y,A,B
			RTS

**
*  FIND NEXT DIRECTORY ENTRY
**
FINDNEXT	PSHU	Y,A,B
			LDB		SECPOS
			CMPB	#$FF-DIRENTSIZE-(256%DIRENTSIZE)
			BLO		_NEXT

			CLR		SECPOS
			LDB		CURSEC
			CMPB	#DIRSECTMAX
			BLO		_NEXTSEC

			ORCC	#1
			PULU	Y,A,B
			RTS

_NEXTSEC	LDA		#DIRTRACK
			INCB					; increment sector
			STB		CURSEC
			JSR		READSECT
			BCC		_NEXTSEC2		; branch if no error
			LDA		#1
			STA		DSKERR
_NEXTSEC2	PULU	Y,A,B
			RTS

_NEXT		ADDB	#DIRENTSIZE
			STB		SECPOS
			LDX		#SECBUF
			CLRA
			LEAX	D,X
			ANDCC	#$FE
			PULU	Y,A,B
			RTS

**
*  READ A DISK SECTOR
*    A = TRACK NUMBER
*    B = SECTOR NUMBER
*  EXIT:
*    A = STATUS CODE
*    X = SECTOR BUFFER
*    Carry set on error
**
READSECT	STA		DCTRK			; track number
			STB		DCSEC			; sector number
			
			if		FLOPPYDRV
			TST		FLOPPY			; floppy drive operation?
			BEQ		READSECT02		; no - skip
			JSR		ENAFLOP			; enable floppy drive
			endif
			
READSECT02	LDX		#SECBUF
			STX		DCBPT

			LDA		#DCRETRY		; load number of retries
			STA		RETRYCNT		; save retries
 
READSECT03			
			* On a DRAGON, the track must be seeked first
			if		DRAGON
			LDA		#1				; seek track
			STA		DCOPC
			JSR		[DSKCON]
			LDA		DCSTA
			BNE		READERR
			endif

			LDA		#2				; read sector
			STA		DCOPC
			JSR		[DSKCON]
			LDA		DCSTA
			BNE		READERR
			ANDCC	#$FE			; clear CC
			
READSECT04	
			if		FLOPPYDRV
			TST		FLOPPY			; floppy drive operation?
			BEQ		READSECT05		; no - skip
			JSR		DISFLOP			; disable floppy drive
			endif
			
READSECT05	LDX		#SECBUF			; load sector buffer
			RTS

READERR		ANDA	#$C0			; keep only bits 6-7 (write protect and drive not ready)
			BNE		READERR02		; yes - do not retry operation
			DEC		RETRYCNT		; decrement retry count
			BNE		READSECT03		; try again
			
READERR02	ORCC	#1
			BRA		READSECT04

**
*  WRITE A DISK SECTOR
*    A = TRACK NUMBER
*    B = SECTOR NUMBER
*  EXIT:
*    A = STATUS CODE
*    X = SECTOR BUFFER
*    Carry set on error
**
WRITSECT	STA		DCTRK			; track number
			STB		DCSEC			; sector number
			
			if		FLOPPYDRV
			TST		FLOPPY			; floppy drive operation?
			BEQ		WRITSECT02		; no - skip
			JSR		ENAFLOP			; enable floppy drive
			endif
			
WRITSECT02	LDX		#SECBUF
			STX		DCBPT

			LDA		#DCRETRY		; load number of retries
			STA		RETRYCNT		; save retries

WRITSECT03
			* On a DRAGON, the track must be seeked first
			if		DRAGON
			LDA		#1				; seek track
			STA		DCOPC
			JSR		[DSKCON]
			LDA		DCSTA
			BNE		WRITERR
			endif

			LDA		#3				; write sector
			STA		DCOPC
			
			JSR		[DSKCON]
			LDA		DCSTA
			BNE		WRITERR
			ANDCC	#$FE

WRITSECT04
			if		FLOPPYDRV
			TST		FLOPPY			; floppy drive operation?
			BEQ		WRITSECT05		; no - skip
			JSR		DISFLOP			; disable floppy drive
			endif
			
WRITSECT05	LDX		#SECBUF			; load sector buffer
			RTS
			
WRITERR		ANDA	#$C0			; keep only bits 6-7 (write protect and drive not ready)
			BNE		WRITERR02		; yes - do not retry operation
			DEC		RETRYCNT		; decrement retry count
			BNE		WRITSECT03		; try again

WRITERR02	ORCC	#1
			BRA		WRITSECT04

**
*  DETECT COCO SDC
*  EXIT:
*    Carry set if COCO SDC is not present
**
DETECTSDC	ORCC	#INTMASKS		; mask irqs
			LDA		#$64			; test pattern
			STA		FLSHDAT			; store at Flash Data Reg address
			LDA		FLSHCTRL		; get value from Flash Ctrl Reg
			CLR		FLSHDAT			; clear Flash Data Reg
			EORA	FLSHCTRL		; get changed bits from Ctrl Reg
			SUBA	#$60			; did expected bits change?
			BNE		DETECTFAIL		; branch if not an SDC
			ANDCC	#$FE			; clear CC
			BRA		DETECTEND

DETECTFAIL	ORCC	#1

DETECTEND	ANDCC	#~INTMASKS		; unmask irqs
			RTS

			if		FLOPPYDRV			
**
*  DETECT THE PRESENCE OF A MULTI-PAK
**
DETECTMPI	LDX		#MPIREG			; get MPI slot sel reg contents
			LDA		#$CC			; load A with %11001100
			ORCC	#INTMASKS		; mask interrupts
			EORA	,X				; xor with MPI slot sel reg contents
			STA		,X				; and save...
			CMPA	,X				; same?
			BNE		DETMPI02		; no - exit
			LDA		#1				; load value
			STA		MPI				; set MPI flag

DETMPI02	ANDCC	#~INTMASKS		; unmask irqs
			RTS

**
*  CHECK THE PRESENCE OF SDC-DOS
*  EXIT:
*    Carry set if SDC-DOS is not present
**
CHKSDCDOS	LDD		#$0203			; SDC-DOS signature
			CMPD	$DFFE			; match?
			BEQ		CHKSDCDOS02		; yes - exit
			LDX		#SDCDOSREQ		; load error message
			JSR		SHOWERR			; display it
			ORCC	#1				; set carry
			RTS
			
CHKSDCDOS02	ANDCC	#$FE			; clear carry
			RTS

**
*  CHECK IF FLOPPY DRIVE OPERATIONS ARE POSSIBLE
*  EXIT:
*    Carry set if floppy drive operations are not possible
**
CHKFLPY		
			if		TEST
			BRA		CHKFLPY03
			endif
			
			JSR		CHKSDCDOS		; check presence of SDC-DOS
			BCS		CHKFLPY02		; no - exit with carry set
			TST		MPI				; is MPI present?
			BNE		CHKFLPY03		; yes - exit with carry clear
			LDX		#MPIREQ			; load error message
			JSR		SHOWERR			; display it

CHKFLPY02	ORCC	#1				; set carry
			RTS
		
CHKFLPY03	ANDCC	#$FE			; clear carry
			RTS	
			endif
			
**
*  LIST THE CURRENT SDC DIRECTORY
**			
LISTSDC
			if		!TEST
			JSR		READSDC			; read the SDC directory
			LBCS	SDCERR			; branch if error
			endif

			TST		MAXFILERR		; check maximum files flag
			BEQ		LISTSDC02		; branch if not set
			LDX		#MAXFLERRMSG	; load maximum files error message
			JSR		SHOWERR			; display error
			
LISTSDC02	LDD		DSKSIZE			; load number of elements
			JSR		SORT			; sort SDC directory

			RTS

**
*  READ THE SDC DIRECTORY
**
READSDC		PSHS	U

			LDD		#0
			STD		DSKSIZE
			STD		DSKSEL
			STD		DSKSELPREV
			STD		DSKOFF
			STA		DIRSIZE
			STA		DIRSEL
			STA		MNTERR
			STA		MNTDRV1
			STA		MAXFILERR
			
			LDD		#DISKARR		; init DISKARR pointer
			STD		PDISKARR

			LDA		#CMDEXD
			CLRB
			LDX		#LISTCMD		; point to command
			JSR		SENDCMD
			BCS		READSDCEND
			
			LDX		PDISKARR
			LDU		#0
			
READSDC01	LDA		#CMDEX
			LDB		#'>'			; set parameter #1
			JSR		SENDCMD
			BCS		READSDCEND

* Check buffer for nulled entry. This signifies the end

			LDA		#16
			LEAY	,X				; go back to start of buffer
READSDC02	LDB		,Y
			BEQ		READSDC03		; null entry?
			LEAU	1,U
			CMPU	#SDCDIRMAX		; maximum number of files?
			BEQ		READSDC02B
			LEAY	16,Y
			DECA
			BNE		READSDC02
			LEAX	256,X			; next sector
			BRA		READSDC01

READSDC02B	LDA		#1
			STA		MAXFILERR		; set error flag

READSDC03	CMPU	#0
			BEQ		READSDC04
			
			LDX		#DISKARR		; check for "." entry
			LDD		,X
			CMPD	#$2E20
			BNE		READSDC04
			
			LEAX	16,X			; remove "." entry
			STX		PDISKARR
			
			LEAU	-1,U

READSDC04	STU		DSKSIZE
			ANDCC	#$FE			
			
READSDCEND	PULS	U
			RTS
			
**
*  CHECK IF THE SELECTED DISK CAN BE MOUNTED
*  EXIT:
*    Carry is set if disk cannot be mounted
**
CHECKDSK	LDD		DSKSIZE			; check if a disk is selected
			BEQ		CHECKFAIL		; no - check failed

			LDD		DSKSEL			; load selected disk
			JSR		GETDSKADR		; get selected disk address

			LDA		11,X			; check file type
			BITA	#$10			; is it a directory?
			BNE		CHECKFAIL		; yes - check failed

			CLRB					; reset carry
			RTS

CHECKFAIL	ORCC	#1
			RTS

**
*  MOUNT THE SELECTED DIRECTORY (MULTI DISKS)
**
MOUNTDIR	LDD		DSKSIZE			; check if a file is selected
			BEQ		MOUNTEND		; no - exit

			LDD		DSKSEL			; load selected disk
			JSR		GETDSKADR		; get selected disk address

			LDA		11,X			; check type
			BITA	#$10			; is a directory?
			BEQ		MOUNTDIREND		; no - exit

			LDA		,X				; check first character
			CMPA	#'.'			; starts with dot?
			BEQ		MOUNTDIREND		; yes - exit

			JSR		MOUNTDSK		; mount directory
			BCS		MOUNTDIREND		; branch on error
			
			JSR		READDIR			; refresh disk directory
			
			LDA		#1				; set multi disk flag
			STA		MULTIDSK

MOUNTDIREND RTS

**
*  MOUNT THE SELECTED DISK
**
MOUNTDSK	LDD		DSKSIZE			; check if a disk is selected
			BEQ		MOUNTEND		; no - exit

			CLR		DIRSIZE
			CLR		OS9DSK
			CLR		BOOTABLE
			CLR		MNTERR
			CLR		MNTDRV1
			CLR		MULTIDSK

			LDY		#STR1			; load string buffer

MOUNT01		LDA		#'M'
			STA		,Y+
			LDA		#':'
			STA		,Y+

			LDD		DSKSEL			; load selected disk
			JSR		GETDSKADR		; get selected disk address
			
			LDD		#11				; copy 11 characters
			JSR		MEMCPY			; copy disk filename
			LEAY	D,Y				; move to end of string
			CLR		,Y				; terminate string

			LDA		#CMDEXD			; extended command with data block
			ADDA	SDCDRV			; set SDC drive number
			CLRB					; clear PARAM1
			LDX		#STR1			; load command string
			JSR		SENDCMD			; send command to the SDC
			BCC		MOUNTEND		; branch on error

			BITB	#$20			; disk already mounted?
			BEQ		MOUNT03			; yes - set mount error flag
			LDA		#1				; set disk mounted in drive 1 flag
			STA		MNTDRV1
			BRA		MOUNTEND		; exit

MOUNT03		LDA		#1				; set mount error flag
			STA		MNTERR

MOUNTEND	RTS

**
*  UNMOUNT THE DISK IN DRIVE SDCDRV
**
UNMOUNT		LDY		#STR1			; copy mount command
			LDA		#'M'			
			STA		,Y+
			LDA		#':'
			STA		,Y+
			CLR		,Y				; terminate string

			LDA		#CMDEXD			; extended command with data block
			ADDA	SDCDRV			; set SDC drive number			
			CLRB					; clear PARAM1
			LDX		#STR1			; load command string
			JSR		SENDCMD			; send command to the SDC
			
UNMOUNTEND	RTS

**
*  CHANGE THE CURRENT DIRECTORY ON THE SDC
**
CHDIR		LDD		DSKSIZE			; check if a file is selected
			BEQ		CHDIREND		; no - exit

			LDD		DSKSEL			; load selected file
			JSR		GETDSKADR		; get file address

			LDA		11,X			; check file type
			BITA	#$10			; is it a directory?
			BEQ		CHDIREND		; no - exit
			
			LDD		,Y				; load filename first 2 characters
			CMPD	#$2E20			; check for current directory entry (. and space)
			BEQ		CHDIREND		; yes - exit
			
			LDY		#STR1			; load command string buffer
			
			LDA		#'D'
			STA		,Y+
			LDA		#':'
			STA		,Y+

			LDD		#11				; copy 11 characters
			JSR		MEMCPY

			LDD		,Y				; load filename
			CMPD	#$2E2E			; check for parent directory entry (..)
			BNE		CHDIR02			; no - skip
			CLR		2,Y				; terminate string

CHDIR02		CLR		11,Y			; terminate string

			LDA		#CMDEXD			; extended command with data block
			CLRB					; clear PARAM1
			LDX		#STR1			; load command string
			JSR		SENDCMD			; send command to the SDC
			BCS		CHDIRERR		; branch on erreur

			JSR		LISTSDC			; refresh SDC directory listing
			JSR		DREAD			; refresh selected disk directory

CHDIREND	RTS

CHDIRERR	JSR		SDCERR
			BRA		CHDIREND

**
*  BOOT THE SELECTED DISK
**
BOOTDSK		LDD		DSKSIZE			; check if a disk is selected
			BEQ		BOOTEND			; no - exit

			LDX		#DOSCMD			; load boot command
			JMP		RUNBAS			; execute
			
BOOTEND		RTS

**
*  CREATE A DISK
**
CREATEDSK	CLR		FILENAME		; initialize filename buffer
			LDX		#CREATELBL		; load window title
			JSR		TEXTBOXWIN		; show window to enter filename
			BCS		CREATEEND		; branch on error

			LDY		#FILENAME		; get new disk filename
			LDX		#DSKEXT			; load disk extension (.DSK)
			JSR		STRCAT			; append to filename
			TFR		Y,X				; transfer filename to X
			JSR		EXPFN			; store in DSKFN
			JSR		DSKINDEX		; get disk index
			BPL		FILEXISTERR		; branch if file already exists	

			LDY		#STR1			; load string buffer
			CLR		,Y				; terminate string
			LDX		#CREATECMD		; load SDC command
			JSR		STRCAT			; append to string buffer

			LDX		#FILENAME		; load new disk filename
			JSR		STRCAT			; append to string buffer

			LDA		#CMDEXD			; extended command with data block
			CLRB					; 0 for single sided 35 tracks
			LDX		#STR1			; load command string
			LDY		#0				; 0 for single sided 35 tracks
			JSR		SENDCMD			; send command to the SDC
			LBCS	SDCERR			; branch on error

			JSR		LISTSDC			; refresh SDC directory

			JSR		DSKINDEX		; get disk index
			BPL		CREATEDSK02		; branch if found
			LDD		#0				; set index to 0
			
CREATEDSK02	STD		DSKSEL			; set selected disk
			JSR		SETDSKOFF		; set disk list offset
			JSR		DREAD			; read selected disk directory
		
CREATEEND	RTS

FILEXISTERR	LDX		#ALREXISTERR	; load error message
			JSR		SHOWERR			; display error
			ORCC	#1				; set carry
			RTS

**
*  MAKE A DIRECTORY
**
MAKEDIR		CLR		FILENAME		; initialize filename buffer
			LDX		#MAKEDIRLBL		; load window title
			JSR		TEXTBOXWIN		; show window to enter filename
			BCS		MAKEDIREND		; branch if user cancels

			LDX		#FILENAME		; load new directory name
			JSR		EXPFN			; store in DSKFN
			JSR		DSKINDEX		; get directory index
			BPL		FILEXISTERR		; branch if directory already exists	

			LDY		#STR1			; load string buffer
			CLR		,Y				; terminate string
			LDX		#MAKEDIRCMD		; load SDC command
			JSR		STRCAT			; append to string buffer

			LDX		#FILENAME		; load new directory name
			JSR		STRCAT			; append to string buffer

			LDA		#CMDEXD			; extended command with data block
			CLRB					; clear PARAM1
			LDX		#STR1			; load command string
			JSR		SENDCMD			; send command to the SDC
			LBCS	SDCERR			; branch on error

			JSR		LISTSDC			; refresh SDC directory

			JSR		DSKINDEX		; get disk index
			BPL		MAKEDIR02		; branch if found
			LDD		#0				; set index to 0
			
MAKEDIR02	STD		DSKSEL			; set selected disk
			JSR		SETDSKOFF		; set disk list offset
			JSR		DREAD			; read selected disk directory

MAKEDIREND	RTS

**
*  REFREH DIRECTORY
**
REFRESH		JSR		READDIR
			RTS

**
*  RENAME DISK OR DIRECTORY
**
RENAME		LDD		DSKSIZE			; check if a disk is selected
			LBEQ	RENAMEEND		; no - exit

			LDD		DSKSEL			; load selected disk
			JSR		GETDSKADR		; get selected disk address
			STX		DSKADR			; save disk address

			LDA		,X				; check for . or ..
			CMPA	#'.'
			LBEQ	RENAMEEND

			LDB		#8				; copy filename to textbox
			LDY		#FILENAME		; load filename address
			CLR		,Y				; clear filename
			JSR		STRNCAT			; copy filename to textbox

			LDX		#FILENAME		; trim filename
			JSR		TRIM

			LDX		DSKADR			; load disk address
			LDA		11,X			; check file type
			BITA	#$10			; is a directory?
			BNE		RENDIR

			LDX		#RENDSKLBL		; load rename disk message
			BRA		RENAME01

RENDIR		LDX		#RENDIRLBL		; load rename directory message

RENAME01	JSR		TEXTBOXWIN		; show window to edit filename
			LBCS	RENAMEEND		; branch if user cancels

			JSR		UNMOUNT			; unmount disk
			LBCS	SDCERR			; branch on error

			LDY		#STR1
			LDA		#'R'
			STA		,Y+
			LDA		#':'
			STA		,Y+

			* COPY OLD FILENAME
			LDX		DSKADR			; load disk address
			LDD		#11				; copy 11 characters
			JSR		MEMCPY			; copy old filename
			LEAY	D,Y				; move to the end of the string
			CLR		,Y+				; add null terminator

			* COPY NEW FILENAME
			LDX		#FILENAME
RENAME02	LDA		,X+
			STA		,Y+
			BNE		RENAME02

			* REMOVE NULL TERMINATOR
			LEAY	-1,Y

			* DOT
			LDA		#'.'
			STA		,Y+

			* COPY EXT FROM OLD FILENAME
			LDX		DSKADR			; load disk address
			LEAX	8,X				; move to the extension
			LDB		#3				; copy 3 characters
RENAME03	LDA		,X+
			STA		,Y+
			DECB
			BNE		RENAME03

			CLR		,Y+				; add null terminator

			LDA		#CMDEXD			; extended command with data block
			CLRB					; clear PARAM1
			LDX		#STR1			; load command string
			JSR		SENDCMD			; send command to the SDC
			BCS		RENAMEERR		; branch on error

			JSR		LISTSDC			; refresh SDC directory
			
			LDX		#STR1+14		; point to filename
			JSR		EXPFN			; store in DSKFN
			JSR		DSKINDEX		; get disk index
			BPL		RENAME04		; branch if disk found
			CLRA					; set index to 0
			CLRB

RENAME04	STD		DSKSEL			; set selected disk
			JSR		SETDSKOFF		; set disk list offset
			JSR		DREAD			; read selected disk directory
			
RENAMEEND	RTS

RENAMEERR	BITB	#$20			; file already exists?
			LBEQ	SDCERR			; no - skip
			JSR		FILEXISTERR		; display file already exists message
			JSR		DREAD			; remount selected disk
			RTS

**
*  DELETE THE SELECTED DISK
**
DELDSK		LDD		DSKSIZE			; check if a disk is selected
			LBEQ	DELEND			; no - exit

			LDD		DSKSEL			; get selected disk
			JSR		GETDSKADR		; get selected disk address
			STX		DSKADR			; save disk address

			LDA		,X				; check for ..
			CMPA	#'.'
			LBEQ	DELEND

			LDD		#$0506			; display confirmation window
			LDX		#23
			LDY		#5
			JSR		WINDOW

			LDX		#CONFIRMLBL		; display confirmation title
			LDD		#$0506
			JSR		PRINTRASC

			LDX		DSKADR			; get disk address
			LDA		11,X			; check type
			BITA	#$10			; is a directory
			BNE		DELDSK02
			LDX		#DELDSKMSG		; load delete disk message
			BRA		DELDSK03

DELDSK02	LDX		#DELDIRMSG		; load delete directory message

DELDSK03	LDD		#$0708			; display confirmation message
			JSR		PRINTASC

			JSR		FLUSHSCR		; update screen

DELDSKKEY	JSR		[POLCAT]
			BEQ		DELDSKKEY

			CMPA	#'Y'
			BEQ		DELDSK04
			CMPA	#'N'
			BEQ		DELEND
			BRA		DELDSKKEY

DELDSK04	JSR		DRAWSCR			; close window
			JSR		UNMOUNT			; unmount disk
			BCS		DELEND			; branch on error

			LDX		DSKADR			; load disk address
			LDY		#STR1			; load string buffer

			LDA		#'X'
			STA		,Y+
			LDA		#':'
			STA		,Y+

			LDD		#11
			JSR		MEMCPY
			LEAY	D,Y

			CLR		,Y				; terminate string

			LDA		#CMDEXD			; extended command with data block
			CLRB					; clear PARAM1
			LDX		#STR1			; load command string
			JSR		SENDCMD			; send command to the SDC
			BCS		DELDSKERR		; branch on error

			LDD 	DSKSEL			; save current disk index
			PSHU	D
			LDD		DSKOFF			; save current disk offset
			PSHU	D

			JSR		LISTSDC			; refresh SDC directory

			PULU	D				; load saved offset
			STD		DSKOFF			; set disk offset
			PULU	D				; load saved index
			CMPD	DSKSIZE			; check if index is still valid
			BLO		DELDSK05
			SUBD	#1

DELDSK05	STD		DSKSEL			; set selected disk

			LDD		DSKSIZE			; check if disk offset needs update
			SUBD	DSKSEL
			CMPD	#WINHEI
			BCC		DELDSK06
			JSR		SETDSK01		; set disk offset			

DELDSK06	JSR		DREAD			; read selected disk directory

DELEND		RTS

DELDSKERR	BITB	#$20			; directory not empty?
			LBEQ	SDCERR			; no - skip
			LDX		#DIRNEMPTERR	; load directory not empty message
			JSR		SHOWERR			; display error message
			JSR		DREAD			; refresh selected disk directory
			RTS

**
*  DISPLAY FILE INFORMATION
**
FILEINFO	LDD		DSKSIZE			; check if a file is selected
			BEQ		FILEINFOEND		; no - exit

			LDD		#$0604
			LDX		#21
			LDY		#7
			JSR		WINDOW

			LDX		#FILEINFOLBL	; load window title
			LDD		#$0604			; display window title
			JSR		PRINTRASC		

			LDX		#NAMELBL		; display name label
			LDD		#$0706
			JSR		PRINTASC

			LDX		#TYPELBL		; display type label
			LDD		#$0707
			JSR		PRINTASC			

			LDX		#SIZELBL		; display size label
			LDD		#$0708
			JSR		PRINTASC			

			LDD		DSKSEL			; display filename
			JSR		GETDSKADR
			TFR		X,Y
			LDX		#STR1
			JSR		FORMATFN
			
			LDD		#$0D06
			JSR		PRINTASC			

			JSR		GETDSKTYPE		; get selected disk type (file/dir)
			LDD		#$0D07
			JSR		PRINTASC
						
			JSR		GETDSKSIZE		; get selected disk size
			LDD		#$0D08
			JSR		PRINTASC
			
			JSR		FLUSHSCR		; refresh screen

FILEINFO02	JSR		[POLCAT]		; wait for a key
			BEQ		FILEINFO02
			
FILEINFOEND	RTS

**
*  SEND A COMMAND TO THE COCO-SDC
*    A = COMMAND CODE
*    B = PARAM1
*    X = COMMAND TEXT
*    Y = PARAM2/PARAM3
**
SENDCMD		PSHS	U,X				; save registers
			TFR		X,U				; U = X
			TFR		Y,X				; X = Y
			JSR		CommSDC			; send command
			PULS	U,X				; restore registers
			
			RTS

*--------------------------------------------------------------------------
* CommSDC
*
*    This is the core routine used for all
*    transactions with the SDC controller.
*
* Entry:
*    A = Command code
*    B = LSN hi byte   or  First parameter byte
*    X = LSN lo word   or  2nd and third parameter bytes
*    U = Address of 256 byte I/O buffer ($FFFF = none)
*
* Exit:
*    Carry set on error.
*    B = controller status code.
*    A, X, Y and U are preserved.
*
CommSDC        pshs      u,y,x,a,cc         ; preserve resgisters
               lsr       ,s                 ; shift carry flag out of saved CC

* Put controller in Command mode
               ldy       #DATREGA           ; setup Y for hardware addressing
               lda       #CMDMODE           ; the magic number
               sta       CTRLATCH           ; send to control latch (FF40)

* Put input parameters into the hardware registers.
* It does no harm to put random data in the
* registers for commands which dont use them.
               stb       -1,y               ; high byte to param reg 1
               stx       ,y                 ; low word to param regs 2 and 3

* Wait for Not Busy.
               bsr      waitForIt           ; run polling loop
               bcs      cmdExit             ; exit if error or timeout

* Send command to controller
               lda      1,s                 ; get preserved command code from stack
               sta      -2,y                ; send to command register (FF48)

* Determine if a data block needs to be sent.
* Any command which requires a data block will
* have bit 5 set in the command code.
               bita      #$20               ; test the "send block" command bit
               beq       rxBlock            ; branch if no block to send

* Wait for Ready to send
               bsr      waitForIt           ; run polling loop
               bcs      cmdExit             ; exit if error or timeout
               leax     ,u                  ; move data address to X

* Send 256 bytes of data
               ldd       #32*256+8          ; 32 chunks of 8 bytes
txChunk        ldu       ,x                 ; send one chunk...
               stu       ,y
               ldu       2,x
               stu       ,y
               ldu       4,x
               stu       ,y
               ldu       6,x
               stu       ,y
               abx                          ; point X at next chunk
               deca                         ; decrement chunk counter
               bne       txChunk            ; loop until all 256 bytes sent

* Wait for command completion
               lda       #5                 ; timeout retries
waitCmplt      bsr       waitForIt          ; run polling loop
               bitb      #BUSY              ; test BUSY bit
               beq       cmdExit            ; exit if completed
               deca                         ; decrement retry counter
               bne       waitCmplt          ; repeat until 0
               coma                         ; set carry for timeout error
               bra       cmdExit            ; exit

* For commands which return a 256 byte response block the
* controller will set the READY bit in the Status register
* when it has the data ready for transfer.   For commands
* which do not return a response block the BUSY bit will
* be cleared to indicate that the command has completed.
*
rxBlock        bsr       longWait           ; run long status polling loop
               bls       cmdExit            ; exit if error, timeout or completed
               leax      1,u                ; test the provided buffer address
               beq       cmdExit            ; exit if "no buffer" ($FFFF)
               leax      ,u                 ; move data address to X

* Read 256 bytes of data
               ldd       #32*256+8          ; 32 chunks of 8 bytes
rxChunk        ldu       ,y                 ; read one chunk...
               stu       ,x
               ldu       ,y
               stu       2,x
               ldu       ,y
               stu       4,x
               ldu       ,y
               stu       6,x
               abx                          ; update X for next chunk
               deca                         ; decrement chunk counter
               bne       rxChunk            ; loop until all 256 bytes transferred
               clrb                         ; status code for SUCCESS, clear carry

* Exit
cmdExit        rol       ,s                 ; rotate carry into saved CC on stack
               clr       CTRLATCH           ; end command mode
               puls      cc,a,x,y,u,pc      ; restore irq masks, update carry and return


*--------------------------------------------------------------------------
* Wait for controller status to indicate either "Not Busy" or "Ready".
* Will time out if neither condition satisfied within a suitable period.
*
* Exit:
*    CC.C set on error or time out.
*    CC.Z set on "Not Busy" status (if carry cleared).
*    B = status
*    X is clobbered.  A, Y and U are preserved.
*
longWait       bsr       waitForIt          ; enter here for doubled timeout
               bcc       waitRet            ; return if cleared in 1st pass
waitForIt      ldx       #0                 ; setup timeout counter
waitLp         comb                         ; set carry for assumed FAIL
               ldb       -2,y               ; read status
               bmi       waitRet            ; return if FAILED
               lsrb                         ; BUSY --> Carry
               bcc       waitDone           ; branch if not busy
               bitb      #READY/2           ; test READY (shifted)
               bne       waitRdy            ; branch if ready for transfer
               bsr       waitRet            ; consume some time
               ldb       #$81               ; status = timeout
               leax      ,-x                ; decrement timeout counter
               beq       waitRet            ; return if timed out
               bra       waitLp             ; try again

waitDone       clrb                         ; Not Busy: status = 0, set Z
waitRdy        rolb                         ; On Ready: clear C and Z
waitRet        rts                          ; return

			if		0						; for debugging purposes
**
*  DSKCON ERROR HANDLER
**
DSKCONERR	LDB		DCSTA
			LDX		#ERRORLBL
			LDY		#STR1
			CLR		,Y
			JSR		STRCAT

			LDX		#STR1
			LEAX	7,X
			CLRA
			JSR		ITOA8
			LDX		#STR1
			
			BRA		SHOWERR
			endif

**
*  SDC ERROR HANDLER
*    B = ERROR CODE
**
SDCERR		CMPB	#$80
			BEQ		SDCBUSY
			BITB	#$20
			BNE		ALROPEN
			BITB	#$10
			BNE		DIRNF
			BITB	#$08
			BNE		HARDWARE
			BITB	#$04
			BNE		INVPATH

			LDX		#UNKNOWNERR
			BRA		SHOWERR

ALROPEN		LDX		#ALROPENERR
			BRA		SHOWERR

DIRNF		LDX		#DIRNFERR
			BRA		SHOWERR

HARDWARE	LDX		#HWERR
			LDA		#1
			STA		FATAL
			BRA		SHOWERR

INVPATH		LDX		#INVPATHERR
			BRA		SHOWERR

SDCBUSY		LDX		#SDCBUSYERR
			BRA		SHOWERR

NOSDC		LDX		#NOSDCERR
			LDA		#1
			STA		FATAL

**
*  DISPLAY ERROR MESSAGE
*    X= ERROR MESSAGE
**
SHOWERR		PSHU	X				; save error message

			LDD		#$0506			; load X/Y coordinates
			LDX		#23				; load width
			LDY		#5				; load height
			JSR		WINDOW			; show window

			LDX		#SDCERRLBL		; load error title
			LDD		#$0506			; load X/Y coordinates
			JSR		PRINTRASC		; show title

			LDD		#$0608			; load X/Y coordinates
			PULU	X				; restore error message
			JSR		PRINTASC		; show error message

			JSR		FLUSHSCR		; refresh screen
			TST		MAXFILERR		; maximum files error?
			BNE		SHOWERREND		; yes - don't wait for a key press

SHOWERRKEY	JSR		[POLCAT]		; read keyboard
			BEQ		SHOWERRKEY		; branch if no key press
			TST		FATAL			; fatal error?
			LBNE	EXIT			; yes, exit application

SHOWERREND	RTS

**
*  TEXTBOX WINDOW
*    X = WINDOW TITLE
*  EXIT:
*    X = TEXT ENTERED
**
TEXTBOXWIN	LDD		#$0806
			JSR		PRINTRASC

TEXTBOX01	LDD		#$0807
			LDX		#16
			LDY		#4
			JSR		WINDOW

			LDD		#$0908
			LDX		#FILENAME
			JSR		PRINTASC

			LDY		#0
			LDX		#FILENAME
COUNTLEN	LDA		,X+
			LEAY	1,Y
			TSTA
			BNE		COUNTLEN
			LEAX	-1,X
			STX		INPUTPOS
			LEAY	$08,Y
			TFR		Y,D
			TFR		B,A
			LDB		#$08
			LDX		#CURSTR
			JSR		PRINTRASC

			JSR		FLUSHSCR

INPUTKEY2	JSR		[POLCAT]
			BEQ		INPUTKEY2
			CMPA	#KEYESC
			BEQ		INPUTCANC

			CMPA	#KEYSLT
			BNE		NOSLEFT

			* SHIFT LEFT
			LDX		#FILENAME
			CLR		,X
			STX		INPUTPOS
			JMP		NEXTINPUT2

NOSLEFT		CMPA	#KEYLT
			BNE		NOLEFT2

			LDD		#FILENAME
			SUBB	INPUTPOS+1
			SBCA	INPUTPOS
			BEQ		NOLEFT2

			LDX		INPUTPOS
			CLR		-1,X
			JMP		NEXTINPUT2
			
NOLEFT2		CMPA	#KEYENT
			BNE		NOENTER2

			* ENTER KEY

			LDD		INPUTPOS		; erase cursor
			SUBD	#FILENAME
			ADDB	#9
			TFR		B,A
			LDB		#$08
			LDX		#CURSTR
			JSR		PRINTASC

			JSR		FLUSHSCR

			LDX		#FILENAME
			CLRB					; clear cc

			JMP		INPUTEND2

NOENTER2	CMPA	#' '
			BLO		NEXTINPUT2
			CMPA	#'_'
			BHI		NEXTINPUT2

			LDX		#INVCHARS
CHKINV		TST		,X
			BEQ		NOENTER3		; char is valid
			CMPA	,X+
			BEQ		NEXTINPUT2		; char is invalid
			BRA		CHKINV

NOENTER3	STA		KEYPRES
			LDD		INPUTPOS
			SUBD	#FILENAME
			CMPB	#8
			BCC		NEXTINPUT2

			LDX		INPUTPOS
			LDA		KEYPRES
			STA		,X+
			CLR		,X

NEXTINPUT2	JMP		TEXTBOX01
INPUTEND2	LEAU	1,U
			RTS

INPUTCANC	ORCC	#1
			BRA		INPUTEND2

**
*  COPY MEMORY
**
MEMCPY		PSHU	X,Y,A,B,CC
			ANDCC	#$AF
			STU		STACKPTR
			TFR		Y,U
			TFR		D,Y
MCLOOP		LDA		,X+
			STA		,U+
			LEAY	-1,Y
			BNE		MCLOOP
			LDU		STACKPTR
			PULU	X,Y,A,B,CC
			RTS

**
*  CLEAR THE SCREEN BUFFER
*    A=CHAR
**
CLRSCR		LDX		#SCRBUF
			LDY		#512
CLS001		STA		,X+
			LEAY	-1,Y
			BNE		CLS001
			CLR		CURPOS
			CLR		CURPOS+1
			RTS

**
*  PRINT ASCII 
*    X=STRING
*    A=COL
*    B=ROW
**
PRINTASC	PSHU	X,Y,A,B,CC
			PSHU	A
			LDA		#32
			MUL
			LDY		#SCRBUF
			LEAY	D,Y
			PULU	A
			LEAY	A,Y
PRAS000		LDA		,X+
			BEQ		PRAS999
			BMI		PRAS002			; semigraphic char
			CMPA	#64
			BGE		PRAS001
			ADDA	#64
			BRA		PRAS002
PRAS001		CMPA	#96
			BLT		PRAS002
			SUBA	#96
PRAS002		STA		,Y+
			BRA		PRAS000
PRAS999		PULU	X,Y,A,B,CC
			RTS

**
*  PRINT REVERSE ASCII
*    X=STRING
*    A=COL
*    B=ROW
**
PRINTRASC	PSHU	X,Y,A,B,CC
			PSHU	A
			LDA		#32
			MUL
			LDY		#SCRBUF
			LEAY	D,Y
			PULU	A
			LEAY	A,Y
PRRA000		LDA		,X+
			BEQ		PRRA999
			BMI		PRRA003			; semi graphic char
			CMPA	#64
			BGE		PRRA001
			BRA		PRRA004
PRRA001		CMPA	#96
			BLT		PRRA002
			SUBA	#96
			BRA		PRRA004
PRRA002		SUBA	#64
			BRA		PRRA004
PRRA003		EORA	#$7F			; invert bits except bit 7
PRRA004		STA		,Y+
			BRA		PRRA000
PRRA999		PULU	X,Y,A,B,CC
			RTS

**
*  PRINT STRING
*    X=CHAR
*    Y=COUNT
*    A=COL
*    B=ROW
**
PRINTSTR	PSHU	Y,X,A
			LDA		#32
			MUL
			LDY		#SCRBUF
			LEAY	D,Y
			PULU	A
			LEAY	A,Y				; Y set to screen buffer
			
			LDB		1,U				; get char
			LDA		3,U				; get count
			LEAU	4,U				; adjust stack
			
PRINTS01	BEQ		PRINTS02
			STB		,Y+
			DECA
			BRA		PRINTS01
			
PRINTS02	RTS

**
*  DRAW WINDOW 
*    A=COL
*    B=ROW
*    X=WIDTH
*    Y=HEIGHT
**
WINDOW		PSHU 	X,Y,A,B,CC
			PSHU	A
			LEAX	-1,X
			STX		WWID
			LEAY	-2,Y
			STY		WHEI
			LDA		#32
			MUL
			LDY		#SCRBUF
			LEAY	D,Y
			PULU	A
			LEAY	A,Y
			STY		WADDR
			LDA		#96
WND001		STA		,Y+
			LEAX	-1,X
			BNE		WND001
			LDA		#141+CLRSCH
			STA		,Y+

WND002		LDY		WADDR
			LEAY	32,Y
			STY		WADDR

			LDX		WWID
			LDA		#96
WND003		STA		,Y+
			LEAX	-1,X
			BNE		WND003

			LDA		#133+CLRSCH
			STA		,Y+

			LDX		WHEI
			LEAX	-1,X
			STX		WHEI
			BNE		WND002

			LDY		WADDR
			LEAY	32,Y
			LDA		#139+CLRSCH
			STA		,Y+
			LDX		WWID
			LDA		#131+CLRSCH
			LEAX	-1,X
WND004		STA		,Y+
			LEAX	-1,X
			BNE		WND004
			LDA		#135+CLRSCH
			STA		,Y+			
			PULU	X,Y,A,B,CC
			RTS
WWID		FDB		0
WHEI		FDB		0
WADDR		FDB		0

**
*  CLEAR RECTANGLE
*    A=COL
*    B=ROW
*    X=WIDTH
*    Y=HEIGHT
**
CLEAR		PSHU 	X,Y,A,B,CC
			PSHU	A
			STX		WWID
			LEAY	-1,Y
			STY		WHEI
			LDA		#32
			MUL
			LDY		#SCRBUF
			LEAY	D,Y
			PULU	A
			LEAY	A,Y
			STY		WADDR
			LDA		#96
CLEAR01		STA		,Y+
			LEAX	-1,X
			BNE		CLEAR01

CLEAR02		LDY		WADDR
			LEAY	32,Y
			STY		WADDR

			LDX		WWID
			LDA		#96
CLEAR03		STA		,Y+
			LEAX	-1,X
			BNE		CLEAR03

			LDX		WHEI
			LEAX	-1,X
			STX		WHEI
			BNE		CLEAR02

			PULU	X,Y,A,B,CC
			RTS

**
*  CLEAR STATUS BAR
**
CLRSTATUS	LDD		#$000F
			LDX		#' '
			LDY		#32
			JSR		PRINTSTR
			RTS
			
**
*  CONCATENATE STRING
*    X=SOURCE STRING
*    Y=DESTINATION STRING
**
STRCAT		PSHU	X,Y,A,B
STRCAT02	TST		,Y+
			BNE		STRCAT02
			LEAY	-1,Y
STRCAT03	LDA		,X+
			STA		,Y+
			BNE		STRCAT03
			PULU	X,Y,A,B
			RTS

**
*  CONCATENATE STRING
*    X=SOURCE STRING
*    Y=DESTINATION STRING
*    B=NUMBER OF CHAR TO COPY
**
STRNCAT		PSHU	X,Y,A,B
STRNCAT02	TST		,Y+
			BNE		STRNCAT02
			LEAY	-1,Y
STRNCAT03	LDA		,X+
			STA		,Y+
			BEQ		STRNCAT04
			DECB
			BNE		STRNCAT03
			CLR		,Y
STRNCAT04	PULU	X,Y,A,B
			RTS

**
*  GET STRING LENGTH
*    X=STRING
*  EXIT:
*    A=STRING LENGTH
**
STRLEN		PSHU	X
			CLRA
			
STRLEN02	TST		,X+
			BEQ		STRLEN03
			INCA
			BRA		STRLEN02
			
STRLEN03	PULU	X
			RTS
			
**
*  TRIM STRING
*    X=STRING
**
TRIM		PSHU	X,Y,A,B
			CLRB
TRIM02		TST		,X
			BEQ		TRIM03
			INCB
			LEAX	1,X
			BRA		TRIM02
TRIM03		LDA		,-X
			CMPA	#' '
			BNE		TRIM04
			DECB
			BEQ		TRIMEND
			BRA		TRIM03
TRIM04		CLR		1,X
TRIMEND		PULU	X,Y,A,B
			RTS

**
*  TRIM LEADING ZEROS FROM STRING
*    X=INPUT STRING
*  EXIT: 
*    X=TRIMMED STRING
**
TRIMZERO	LDA		,X				; load character from string
			CMPA	#'0'			; character zero?
			BNE		TRIMZ02			; no - exit
			LEAX	1,X				; increment X
			BRA		TRIMZERO		; continue until done
			
TRIMZ02		RTS
			
**
*  STRNCMP - Compare two strings
*    X = First string to compare
*    Y = Second string to compare
*    B = Number of characters to compare
*  EXIT:
*    B= 0 strings are equals
*    B=-1 first string is lower than second string
*    B= 1 first string is higher than second string
**
STRNCMP		PSHS	X,Y,A

STRNC01		LDA		,X+
			CMPA	,Y+
			BHI		STRNHI
			BLO		STRNLOW
			CMPA	#0
			BEQ		STRNEQU
			DECB
			BEQ		STRNEQU
			BRA		STRNC01

STRNLOW		LDB		#-1
			PULS	X,Y,A
			RTS

STRNHI		LDB		#1
			PULS	X,Y,A
			RTS

STRNEQU		CLRB
			PULS	X,Y,A
			RTS

**
*  ITOA
*    D=NUMBER
*    X=STRING
**			
ITOA8		PSHU	X,Y,A,B,CC
			JSR		ITOA004
			PULU	X,Y,A,B,CC
			RTS
**
*  ITOA
*    D=NUMBER
*    X=STRING
**
ITOA16		PSHU	X,Y,A,B,CC
			JSR		ITOA003
			PULU	X,Y,A,B,CC
			RTS

ITOA003		LDY 	#10000
			JSR		ITOA000
			LDY 	#1000
			JSR		ITOA000
ITOA004		LDY		#100
			JSR		ITOA000
			LDY		#10
			JSR		ITOA000
			LDY		#1
ITOA000		STD		NUMBER
			STY		DIGIT
			LDA		#'0'
			STA		,X
ITOA001		LDD		NUMBER
			SUBD	DIGIT
			BCS		ITOA002
			STD		NUMBER
			LDA		,X
			INCA
			STA		,X
			JMP		ITOA001
ITOA002		CLRA
			LEAX	1,X
			STA		,X
			LDD 	NUMBER
			RTS

NUMBER		FDB		0
DIGIT		FDB		0

**
*  SORT - Sort disk listing with the Heap Sort algorithm
*    D = Number of elements
**
SORT		CMPD	#0
			LBEQ	SORTEND

			CMPD	#50				; display message if dsksize >= 50
			BLO		SORT02

			PSHS	A,B

			JSR		CLRSTATUS		; clear status bar
			
			LDX		#SORTLBL
			LDD		#$010F
			JSR		PRINTRASC

			JSR		FLUSHSCR		; update screen

			PULS	A,B

SORT02		SUBD	#1				; last=dsksize-1
			BEQ		SORTEND
			STD		LAST

			LDX		PDISKARR		
			PSHU	X				; save PDISKARR on the stack
			LDY		,X				; check for .. entry
			CMPY	#$2E2E			; ".."
			BNE		SORT03
			
			LEAX	16,X			; Remove .. from sorting
			STX		PDISKARR
			
			SUBD	#1
			STD		LAST
			
SORT03		ADDD	#1
			LSRA					; divide D by 2
			RORB

SORT04		STD		CTR				; ctr=dsksize/2
			STD		PARENT
			JSR		MAXHEAP
			LDD		CTR				; ctr--
			SUBD	#1
			BPL		SORT04			; ctr >= 0?
			LDD		LAST			; ctr=dsksize-1
			BEQ		SORT06			; ctr == 0?

SORT05		STD		CTR
			JSR		GETDSKADR		; x=diskarr[i]
			LDY		PDISKARR		; y=diskarr[0]
			JSR		SWAP
			LDD		#0				; parent=0
			STD		PARENT
			LDD		CTR				; last=ctr-1
			SUBD	#1
			STD		LAST
			JSR		MAXHEAP
			LDD		CTR				; ctr--
			SUBD	#1
			BNE		SORT05

SORT06		PULU	X				; restore PDISKADR
			STX		PDISKARR

SORTEND		RTS

**
*  MAXIMUM HEAP SUBROUTINE FOR SORT
**
MAXHEAP		LDD		PARENT
			LSLB					; multiply by 2
			ROLA
			STD		CHILD			; child=2*parent
			CMPD	LAST
			BHI		MAXHEAP04		; child > last?

MAXHEAP02	BHS		MAXHEAP03		; child >= last?
			JSR		GETDSKADR		; x=diskarr[child]
			LEAY	16,X			; y=diskarr[child+1]
			JSR		CMPDISK
			BHS		MAXHEAP03
			LDD		CHILD			; child++
			ADDD	#1
			STD		CHILD

MAXHEAP03	LDD		CHILD
			JSR		GETDSKADR		; x=diskarr[child]
			TFR		X,Y
			LDD		PARENT
			JSR		GETDSKADR		; x=diskarr[parent]
			JSR		CMPDISK
			BHS		MAXHEAP04
			JSR		SWAP			; swap diskarr[parent] and diskarr[child]
			LDD		CHILD			; parent=child
			STD		PARENT
			LSLB					; multiply by 2
			ROLA
			STD		CHILD			; child=2*parent
			CMPD	LAST
			BLE		MAXHEAP02		; child <= last?

MAXHEAP04	RTS

**
*  CMPDISK
*    X = FIRST DISK
*    Y = SECOND DISK
**
CMPDISK 	LDB		11,X			; file attributes x
			ANDB	#$10			; keep dir bit
			STB		DIRBIT			; store
			LDB		11,Y			; file attributes y
			ANDB	#$10			; keep dir bit
			CMPB	DIRBIT
			BNE		CMPDISK02
			LDB		#11				; compare 11 characters
			JSR		STRNCMP

CMPDISK02	RTS

**
*  SWAP TWO 16 BYTES ELEMENTS
*    X = FIRST ELEMENT
*    Y = SECOND ELEMENT
**
SWAP		LEAS	-1,S			; reserve 1 byte for counter
			LDB		#16
			STB		,S

SWAP02		LDA		,X				; load char of first string
			LDB		,Y				; load char of second string
			STA		,Y+				; save char in second string
			STB		,X+				; save char in first string
			DEC		,S				; decrement counter
			BNE		SWAP02

			LEAS	1,S				; restore stack
			RTS

**
*  DSORT - Sort directory listing with the Heap Sort algorithm
*    D=Number of elements
**
DSORT		CMPD	#0
			BEQ		DSORTEND

			CMPD	#50				; display message if dirsize >= 50
			BLO		DSORT02

			PSHS	A,B

			JSR		CLRSTATUS		; clear status bar
			
			LDX		#SORTLBL
			LDD		#$010F
			JSR		PRINTRASC

			JSR		FLUSHSCR

			PULS	A,B

DSORT02		SUBD	#1				; last=dirsize-1
			STD		LAST

			ADDD	#1
			LSRA					; divide D by 2
			RORB

DSORT03		STD		CTR				; ctr=dirsize/2
			STD		PARENT
			JSR		DMAXHEAP
			LDD		CTR				; ctr--
			SUBD	#1
			BPL		DSORT03			; ctr >= 0?
			LDD		LAST			; ctr=dirsize-1
			BEQ		DSORTEND		; ctr == 0?

DSORT04		STD		CTR
			JSR		GETDIRADR		; x=dirarr[i]
			LDY		#DIRARR			; y=dirarr[0]
			JSR		SWAP
			LDD		#0				; parent=0
			STD		PARENT
			LDD		CTR				; last=ctr-1
			SUBD	#1
			STD		LAST
			JSR		DMAXHEAP
			LDD		CTR				; ctr--
			SUBD	#1
			BNE		DSORT04

DSORTEND	RTS

**
*  MAXIMUM HEAP SUBROUTINE FOR DSORT
**
DMAXHEAP	LDD		PARENT
			LSLB					; multiply by 2
			ROLA
			STD		CHILD			; child=2*parent
			CMPD	LAST
			BHI		DMAXHEAP04		; child > last?

DMAXHEAP02	BHS		DMAXHEAP03		; child >= last?
			JSR		GETDIRADR		; x=dirarr[child]
			LEAY	16,X			; y=dirarr[child+1]
			LDB		#12				; 12 characters
			JSR		STRNCMP			; compare filename
			BHS		DMAXHEAP03
			LDD		CHILD			; child++
			ADDD	#1
			STD		CHILD

DMAXHEAP03	LDD		CHILD
			JSR		GETDIRADR		; x=dirarr[child]
			TFR		X,Y
			LDD		PARENT
			JSR		GETDIRADR		; x=dirarr[parent]
			LDB		#12				; 12 characters
			JSR		STRNCMP			; compare filename
			BHS		DMAXHEAP04
			JSR		SWAP			; Swap dirarr[parent] and dirarr[child]
			LDD		CHILD			; parent=child
			STD		PARENT
			LSLB					; multiply by 2
			ROLA
			STD		CHILD			; child=2*parent
			CMPD	LAST
			BLE		DMAXHEAP02		; child <= last?

DMAXHEAP04	RTS

**
*  GET DIR ENTRY ADDRESS
*    B = DIR ENTRY NUMBER
*  EXIT:
*    X = DIR ENTRY ADDRESS
**		
GETDIRADR	LDA		#16
			MUL

			LDX		#DIRARR
			LEAX	D,X

			RTS

**
*  FORMAT DIR FILENAME
*    X = DESTINATION STRING
*    Y = SOURCE STRING
**
FORMATDFN	PSHU	X,Y

			LDA		#' '			; add a space
			STA		,X+

			LDB		#12				; copy filename, dot and extension
			
FORMATD02	LDA		,Y+				; load source char
			STA		,X+				; save char to destination
			DECB					; decrement character count
			BNE		FORMATD02		; branch if not done

			LDA		#' '			; add a space
			STA		,X+
			
			CLR		,X				; terminate string
			
FORMATD03	PULU	X,Y
			RTS			
	
			if		FLOPPYDRV
**
*  DISPLAY FLOPPY DISK WINDOW
*	 X=WINDOW TITLE
*  EXIT:
*    Carry is set if user aborts operation
**
FLOPPYWIN	PSHU	X				; save window title on stack
			JSR		DRAWBG			; draw background

			LDD		#$0303
			LDX		#27
			LDY		#10
			JSR		WINDOW

			PULU	X				; get window title from stack
			LDD		#$0303			; display window title
			JSR		PRINTRASC		

			TST		FORMATWIN		; is it format window?
			BNE		FLOPPYW02		; yes - skip		
			
			LDX		#FILENAMELBL	; display filename label
			LDD		#$0405
			JSR		PRINTASC

			LDD		DSKSEL			; display filename
			JSR		GETDSKADR
			TFR		X,Y
			LDX		#STR1
			JSR		FORMATFN

			LDD		#$0E05
			JSR		PRINTASC

FLOPPYW02	LDX		#TRACKSLBL		; display tracks label
			LDD		#$0406
			SUBB	FORMATWIN		; adjust height for format window
			JSR		PRINTASC

			LDX		#SIDELBL		; display side label
			LDD		#$0407
			SUBB	FORMATWIN		; adjust height for format window
			JSR		PRINTASC
			
			TST		FORMATWIN		; is it format window?
			BEQ		FLOPPYW03		; no - skip				

			LDX		#VERIFYLBL		; display verify label
			LDD		#$0407
			JSR		PRINTASC
			
FLOPPYW03	LDX		#FLPYMSG1		; insert disk message
			LDD		#$0409
			JSR		PRINTASC

			LDX		#FLPYMSG2		; press enter when ready message
			LDD		#$040A
			JSR		PRINTASC

			JSR		CLRSTATUS		; clear status bar

			LDX		#EXITCMD		; display exit command
			LDD		#$010F
			JSR		PRINTRASC
			
			LDX		#DRVSELCMD		; display floppy drive command
			LDD		#$160F
			JSR		PRINTRASC

FLOPPYW04	LDA		FDNUM			; load floppy drive number
			CMPA	#3				; drive 3?
			BNE		FLOPPYW05		; no - skip
			TST		FLPYSIDE		; double sided disk?
			BEQ		FLOPPYW05		; no - skip
			CLRA					; set drive number to 0
			STA		FDNUM			; store value

FLOPPYW05	ADDA	#'0'			; convert to ASCII
			STA		SCRBUF+9*32+25	; display value
			
			CLRA					; display number of tracks
			LDB		FLPYTRK			; load number of tracks
			LDX		#STR1			; initialize string buffer
			JSR		ITOA8			; convert to ASCII

			LDD		#$0E06			; load X/Y coordinates
			SUBB	FORMATWIN		; adjust height for format window
			LEAX	1,X				; skip first digit
			JSR		PRINTASC		; display value

			LDA		FLPYSIDE		; display side selection
			CMPA	#1				; side 2?
			BEQ		FLOPPYW06		; yes - branch
			CMPA	#2				; double sided?
			BEQ		FLOPPYW07		; yes branch
			LDX		#SSLBL			; single sided label
			BRA		FLOPPYW08
			
FLOPPYW06	LDX		#S2LBL			; side 2 label
			BRA		FLOPPYW08
			
FLOPPYW07	LDX		#DSLBL			; double sided label
FLOPPYW08	LDD		#$0E07			; load X/Y coordinates
			SUBB	FORMATWIN		; adjust height for format window
			JSR		PRINTASC		; display value

			TST		FORMATWIN		; is it format window?
			BEQ		FLOPPYW11		; no - skip
			
			LDA		FLPYVER			; load floppy verification flag
			BEQ		FLOPPYW09		; no - skip
			LDX		#YESLBL			; load yes label
			BRA		FLOPPYW10

FLOPPYW09	LDX		#NOLBL			; load No label
FLOPPYW10	LDD		#$0E07			; load X/Y coordinates
			JSR		PRINTASC		; display value
		
FLOPPYW11	JSR		FLUSHSCR		; refresh screen

			JSR		FLOPPYKEY		; wait for a key

			CMPA	#KEYESC			; check of BREAK key
			BEQ		FLOPPYEND

			CMPA	#KEYENT			; check for ENTER key
			BNE		FLOPPYW04		; no - refresh options

			LDD		#$0308			; clear window
			LDX		#26
			LDY		#4
			JSR		CLEAR

			JSR		CLRSTATUS		; clear status bar
			
			LDX		#ABORTCMD		; display abort command
			LDD		#$010F
			JSR		PRINTRASC

			JSR		FLUSHSCR		; update screen

			ANDCC	#$FE			; clear carry
			RTS

FLOPPYEND	ORCC	#1				; set carry
			RTS

**
*  HANDLE FLOPPY DISK WINDOW COMMANDS
**
FLOPPYKEY	JSR		[POLCAT]		; read keyboard
			BEQ		FLOPPYKEY		; loop until a key is pressed

			PSHU	A				; save key pressed on stack
			
			CMPA	#KEYESC			; BREAK key
			BEQ		FLOPPYKEND		; exit
			
			CMPA	#KEYENT			; ENTER key
			BEQ		FLOPPYKEND		; exit

			CMPA	#'0'			; 0-3
			BLO		FLOPPYK02
			CMPA	#'0'+FLPYMAX
			BHI		FLOPPYK02
			SUBA	#'0'			; convert ASCII to int
			STA		FDNUM			; set floppy drive number
			BRA		FLOPPYKEND		; refresh floppy drive number

FLOPPYK02	CMPA	#'T'			; check for T key
			BNE		FLOPPYK06		; no - skip
			LDA		FLPYTRK			; load number of tracks
			CMPA	#35				; is 35?
			BNE		FLOPPYK03		; no - skip
			LDA		#40				; 40 tracks
			BRA		FLOPPYK05		; set number of tracks
FLOPPYK03	CMPA	#40				; is 40?
			BNE		FLOPPYK04		; no - skip
			LDA		#80				; 80 tracks
			BRA		FLOPPYK05		; set number of tracks
FLOPPYK04	LDA		#35				; 35 tracks
FLOPPYK05	STA		FLPYTRK			; store value
			BRA		FLOPPYKEND		; exit

FLOPPYK06	CMPA	#'S'			; check for S key
			BNE		FLOPPYK08		; no - skip
			LDA		FLPYSIDE		; load side selection
			INCA					; increment
			CMPA	#2				; greater than 2?
			BLS		FLOPPYK07		; no - skip
			CLRA					; reset value
FLOPPYK07	STA		FLPYSIDE		; store value
			BRA		FLOPPYKEND		; refresh side selection
			
FLOPPYK08	TST		FORMATWIN		; is it format window?
			BEQ		FLOPPYK09		; no - skip
			CMPA	#'V'			; check for V key
			BNE		FLOPPYK09		; no - skip
			LDA		FLPYVER			; load verify flag
			INCA					; increment
			ANDA	#1				; keep only bit 0
			STA		FLPYVER			; store
			BRA		FLOPPYKEND		; refresh value

FLOPPYK09	BRA		FLOPPYKEY		; read keyboard again

FLOPPYKEND	PULU	A				; restore key pressed
			RTS

**
*  FLOPPY DISK DIRECTORY
**
FLOPPYDIR	CLR		SIDE			; reset side number
			JSR		DRAWBG			; draw background

FLOPPYD02	LDD		#$0202			; display window
			LDX		#29
			LDY		#12
			JSR		WINDOW

			LDX		#FLPYDIRLBL 	; get window title
			LDD		#$0202			; display window title
			JSR		PRINTRASC

			LDA		FDNUM			; load floppy drive number
			CMPA	#3				; drive 3?
			BNE		FLOPPYD03		; no - skip
			CLR		SIDE			; reset side number
			
FLOPPYD03	ADDA	#'0'			; convert to ASCII
			STA		SCRBUF+2*32+22	; display value

			JSR		CLRSTATUS		; clear status bar

			LDX		#SIDESELCMD		; display side command
			LDD		#$010F
			JSR		PRINTRASC
			
			LDA		SIDE			; load side number
			INCA					; increment to get values 1-2
			ADDA	#'0'			; convert to ASCII
			STA		SCRBUF+15*32+8	; display value
			
			LDX		#DRVSELCMD		; display floppy drive commands
			LDD		#$160F
			JSR		PRINTRASC

			LDX		#FLPYREADLBL	; display loading message
			LDD		#$0604
			JSR		PRINTASC

			JSR		FLUSHSCR		; update screen

			LDA		#1				; enable floppy drive
			STA		FLOPPY
			JSR		READDIR			; read directory
			LDA		#0				; disable floppy drive
			STA		FLOPPY
			
			TST		DSKERR			; check for error
			BNE		FLOPPYD04		; yes - skip
			
			LDA		DIRSIZE			; load directory size
			CMPA	#20				; more than 20 entries?
			BLS		FLOPPYD04		; no - skip
			
			LDX		#UPDOWNCMD		; display up/down commands
			LDD		#$0C0F
			JSR		PRINTRASC
			
FLOPPYD04	JSR		LISTFLPY		; list floppy drive files
			JSR		FLUSHSCR		; update screen

FLOPPYDKEY	JSR		[POLCAT]		; wait for a key
			BEQ		FLOPPYDKEY
			CMPA	#KEYESC			; BREAK key
			BEQ		FLOPPYDEND
			CMPA	#' '			; SPACE key
			BEQ		FLOPPYDEND
			CMPA	#KEYENT			; ENTER key
			BEQ		FLOPPYDEND

			CMPA	#'0'			; 0-3
			BLO		FLOPPYD05
			CMPA	#'0'+FLPYMAX
			BHI		FLOPPYD05
			SUBA	#'0'			; convert ASCII to int
			STA		FDNUM			; set floppy drive number
			LBRA	FLOPPYD02		; refresh directory

FLOPPYD05	CMPA	#KEYUP			; UP arrow key
			BNE		FLOPPYD06

			LDA		DIROFF			; load directory offset
			BEQ		FLOPPYDKEY		; already at top of the list - exit
			SUBA	#20				; decrement directory offset by 20
			STA		DIROFF			; store value
			BRA		FLOPPYD04		; update screen

FLOPPYD06	CMPA	#KEYDN			; DOWN arrow key
			BNE		FLOPPYD07

			LDA		DIROFF			; load directory offset
			ADDA	#20				; decrement directory offset by 20
			CMPA	DIRSIZE			; already at bottom of the list?
			BHS		FLOPPYDKEY		; yes - exit
			STA		DIROFF			; store value
			BRA		FLOPPYD04		; update screen			

FLOPPYD07	CMPA	#'S'			; S key
			BNE		FLOPPYDKEY
			LDA		SIDE			; get current side
			COMA					; flip bits
			ANDA	#1				; keep only bit 0
			STA		SIDE			; store value
			BEQ		FLOPPYD08		; branch if not second side
			JSR		CHKALLRAM		; ensure ALL RAM mode
			
FLOPPYD08	LBRA	FLOPPYD02		; refresh directory	
					
FLOPPYDEND	RTS

**
*  DISPLAY FLOPPY DISK DIRECTORY
**
LISTFLPY	LDD		#$0203			; clear window
			LDX		#28
			LDY		#10
			JSR		CLEAR

			TST		DSKERR			; check for error
			BNE		LISTFLPY03
			
			TST		OS9DSK			; check for OS-9 disk
			BNE		LISTFLPY04

			LDB		DIRSIZE			; number files on floppy disk
			BEQ		LISTFLEND		; exit if no files
			SUBB	DIROFF
			CMPB	#20				; maximum 20 files
			BLO		LISTFLPY01
			LDB		#20

LISTFLPY01	LDY		#0
			LEAY	B,Y

			LDB		#3
			STB		CURLINE			; start line

			LDB		DIROFF
			JSR		GETDIRADR		; get directory address

			* FIRST COLUMN

LISTFLPY02	PSHU	X,Y

			TFR		X,Y
			LDX		#STR1			; get filename
			JSR		FORMATDFN		 

			LDA		#2
			LDB		CURLINE			; curr line
			JSR		PRINTASC

			PULU	X,Y

			LEAX	16,X			; next file
			LEAY	-1,Y

			BEQ		LISTFLEND		; end of list

			* SECOND COLUMN

			PSHU	X,Y

			TFR		X,Y
			LDX		#STR1			; get filename
			JSR		FORMATDFN		 

			LDA		#16
			LDB		CURLINE			; curr line
			JSR		PRINTASC

			PULU	X,Y

			INC		CURLINE			; next line
			LEAX	16,X			; next file
			LEAY	-1,Y

			BNE		LISTFLPY02

LISTFLEND	RTS		

LISTFLPY03	LDD		#$0704			; display error message
			LDX		#READDSKERR
			JSR		PRINTASC
			BRA		LISTFLEND

LISTFLPY04	LDD		#$0C04			; display OS-9 disk message
			LDX		#OS9DSKMSG
			JSR		PRINTASC
			BRA		LISTFLEND

**
*  READ FLOPPY DISK
**
READFLPY	LDX		#READFLPYLBL	; get window title
			JSR		FLOPPYWIN		; display floppy disk window
			LBCS	READFLEND		; exit - user aborted operation

			LDX		#FLPYINFOLBL	; display track/sector labels
			LDD		#$0409
			JSR		PRINTASC

			JSR		FLUSHSCR		; update screen
			
			JSR		CHKALLRAM		; ensure ALL RAM mode

			CLR		SIDE			; start at side 0
			CLR		ERRCTR			; reset error count
			
			LDA		FLPYSIDE		; get side selection
			CMPA	#1				; single sided side 2?
			BNE		READFL02		; no - skip
			INC		SIDE			; increment side number

READFL02	CLR		TRACK			; start at track 0

READFL03	LDA		#1				; start at sector 1
			STA		SECT

READFL04	JSR		TRACKINFO		; update track/sector information

			JSR		[POLCAT]		; check for BREAK key
			CMPA	#KEYESC
			BEQ		READFL11

			* read sector

			LDA		#1
			STA		FLOPPY			; enable floppy drive
			LDA		TRACK			; track number
			LDB		SECT			; sector number
			JSR		READSECT		; read sector
			LDA		#0
			STA		FLOPPY			; disable floppy drive
			BCC		READFL05		; branch if no error

			LDX		#READERRLBL		; load error label
			JSR		FLOPPYERR		; display error count
			BCS		READFL09B		; branch on fatal error
			BRA		READFL06		; do not write sector

			* write sector

READFL05	LDA		TRACK			; track number
			LDB		SECT			; sector number
			JSR		WRITELSN		; write LSN sector
			BCS		READFLERR		; branch on error

READFL06	LDA		SECT			; increment sector number
			INCA
			CMPA	#FLPYSECT
			BHI		READFL07
			STA		SECT
			BRA		READFL04		; copy next sector

READFL07	LDA		TRACK			; increment track number
			INCA
			CMPA	FLPYTRK
			BEQ		READFL08
			STA		TRACK
			BRA		READFL03		; copy next track

READFL08	LDA		FLPYSIDE		; load side selection
			CMPA	#2				; double sided disk?
			BNE		READFL09		; no - skip
			LDA		SIDE			; load side number
			INCA					; increment
			CMPA	#1				; last side?
			BHI		READFL09		; yes - exit
			STA		SIDE			; store value
			BRA		READFL02		; copy next side
			
READFL09	LDX		#OPCOMPLBL		; display operation completed
			LDD		#$0409
			JSR		PRINTASC

READFL09B	JSR		CLRSTATUS		; clear status bar

			LDX		#ANYKEYCMD		; display press any key message
			LDD		#$010F
			JSR		PRINTRASC

			JSR		FLUSHSCR		; update screen

			JSR		BEEP			; play sound

READFL10	JSR		[POLCAT]		; wait for a key
			BEQ		READFL10

READFL11	JSR		DREAD			; refresh directory

READFLEND	RTS

READFLERR	LDX		#WRITERRMSG
			JSR		SHOWERR
			RTS

**
*  WRITE FLOPPY DISK
**
WRITEFLPY	LDX		#WRITFLPYLBL	; get window title
			JSR		FLOPPYWIN		; display floppy disk window
			LBCS	WRITEFLEND		; exit - user aborted operation

			LDX		#FLPYINFOLBL	; display track/sector labels
			LDD		#$0409
			JSR		PRINTASC

			JSR		FLUSHSCR		; update screen
			
			JSR		CHKALLRAM		; ensure ALL RAM mode
			
			CLR		SIDE			; start at side 0
			CLR		ERRCTR			; reset error count

			LDA		FLPYSIDE		; get side selection
			CMPA	#1				; single sided side 2?
			BNE		WRITEFL02		; no - skip
			INC		SIDE			; increment side number
			
WRITEFL02	CLR		TRACK			; start at track 0

WRITEFL03	LDA		#1				; start at sector 1
			STA		SECT

WRITEFL04	JSR		TRACKINFO		; update track/sector information

			JSR		[POLCAT]		; check for BREAK key
			CMPA	#KEYESC
			BEQ		WRITEFLEND

			* read sector

			LDA		TRACK			; track number
			LDB		SECT			; sector number
			JSR		READLSN			; read LSN sector
			BCS		WRITEFLERR		; branch on error

			* write sector

			LDA		#1
			STA		FLOPPY			; enable floppy drive
			LDA		TRACK			; track number
			LDB		SECT			; sector number
			JSR		WRITSECT		; write sector
			LDA		#0
			STA		FLOPPY			; disable floppy drive
			BCC		WRITEFL05		; branch if no error

			LDX		#WRITERRLBL		; load error label
			JSR		FLOPPYERR		; display error count
			BCS		WRITEFL08B		; branch on fatal error

WRITEFL05	LDA		SECT			; increment sector number
			INCA
			CMPA	#FLPYSECT
			BHI		WRITEFL06
			STA		SECT
			BRA		WRITEFL04		; copy next sector

WRITEFL06	LDA		TRACK			; increment track number
			INCA
			CMPA	FLPYTRK
			BEQ		WRITEFL07
			STA		TRACK
			BRA		WRITEFL03		; copy next track

WRITEFL07	LDA		FLPYSIDE		; load side selection
			CMPA	#2				; double sided disk?
			BNE		WRITEFL08		; no - skip
			LDA		SIDE			; load side number
			INCA					; increment
			CMPA	#1				; last side?
			BHI		WRITEFL08		; no - exit
			STA		SIDE			; store value
			BRA		WRITEFL02		; copy next side	

WRITEFL08	LDX		#OPCOMPLBL		; display operation completed
			LDD		#$0409
			JSR		PRINTASC

WRITEFL08B	JSR		CLRSTATUS		; clear status bar

			LDX		#ANYKEYCMD		; display press any key message
			LDD		#$010F
			JSR		PRINTRASC

			JSR		FLUSHSCR		; update screen

			JSR		BEEP			; play sound

WRITEFL09	JSR		[POLCAT]		; wait for a key
			BEQ		WRITEFL09

WRITEFLEND	RTS

WRITEFLERR	LDX		#READERRMSG
			JSR		SHOWERR
			RTS

**
*  DISPLAY TRACK/SECTOR INFORMATION
**
TRACKINFO	CLRA					; display track number
			LDB		TRACK
			LDX		#STR1
			JSR		ITOA8

			LEAX	1,X
			LDD		#$0A09
			JSR		PRINTASC

			CLRA					; display sector number
			LDB		SECT
			LDX		#STR1
			JSR		ITOA8

			LEAX	1,X
			LDD		#$1609
			JSR		PRINTASC

			JSR		FLUSHSCR		; refresh screen

			RTS

**
*  DISPLAY FLOPPY READ/WRITE ERRORS
*    X=ERROR LABEL
*    EXIT: carry set on fatal error
**
FLOPPYERR	LDB		DCSTA			; load status
			ANDB	#$C0			; keep only bits 6-7 (write protect and drive not ready)
			BEQ		FLOPPYERR02		; branch on non fatal error
			
			JSR		SHOWERRMSG		; show error message
			JSR		FLUSHSCR		; update screen
			ORCC	#1				; fatal error - set carry
			RTS

FLOPPYERR02	LDD		#$040A			; display error label
			JSR		PRINTASC			

			JSR		STRLEN			; get string length
			PSHU	A				; save string length

			INC		ERRCTR			; increment error count

			CLRA					; load error count			
			LDB		ERRCTR
			LDX		#STR1			; convert to ASCII
			JSR		ITOA8			

			LEAX	1,X				; display error count
			PULU	A				; restore string length
			ADDA	#5
			LDB		#10
			JSR		PRINTASC

			JSR		FLUSHSCR		; update screen
			CLRA					; clear carry
			RTS

**
*  FORMAT FLOPPY DISK
**
FORMATFLPY	LDA		#1				; set FORMATWIN flag
			STA		FORMATWIN
			LDX		#FMTFLPYLBL		; get window title
			JSR		FLOPPYWIN		; display floppy disk window
			LDA		#0				; clear FORMATWIN flag
			STA		FORMATWIN
			LBCS	FORMATFLEND		; exit - user aborted operation

			LDX		#FMTINFOLBL		; display formatting label
			LDD		#$0409
			JSR		PRINTASC

			JSR		FLUSHSCR		; update screen
			
			JSR		CHKALLRAM		; ensure ALL RAM mode

			CLR		SIDE			; start at side 0
			CLR		ERRCTR			; reset error count

			CLR		TRACK

			LDA		FLPYSIDE		; get side selection
			CMPA	#2				; double sided disk?
			BNE		FORMATFL02		; no - skip
			LDB		#3				; default skip factor (OS-9)
			BRA		FORMATFL02C
			
FORMATFL02	CMPA	#1				; single sided side 2?
			BNE		FORMATFL02B		; no - skip
			INC		SIDE			; increment side number
			
FORMATFL02B	LDB		#4				; default skip factor (RS-DOS)

FORMATFL02C	STU		UREG			; save U register
			
									; FROM DISK BASIC 1.1 DISSASEMBLY - $D5B2
									
			PSHS	B 				; save skip factor on the stack
			LDX		#DBUF1+FLPYSECT	; point to end of logical sector number storage area
			LDB		#FLPYSECT		; load number of sectors per track
FORMATFL03	CLR		,-X				; clear a byte in the buffer
			DECB					; cleared all 18?
			BNE		FORMATFL03		; keep going if not
			CLRA					; reset physical sector counter
			BRA		FORMATFL07		; start with first physical sector = 1
			
									; calculate logical sector numbers

FORMATFL04 	ADDB	,S				; add skip factor to logical sector counter
FORMATFL05	INCB 					; add one to logical sector counter
FORMATFL06 	SUBB	#FLPYSECT		; subtract max number of sectors
			BHS		FORMATFL06		; branch until 0 > ACCB >= -18
			ADDB	#FLPYSECT		; add 18, now accb is 0-17
			TST		B,X				; is anything stored here already?
			BNE		FORMATFL05		; yes - get another sector

FORMATFL07	INCA					; * increment physical sector number and
			STA		B,X				; * save it in the ram buffer
			CMPA	#FLPYSECT		; finished with all sectors?
			BLO		FORMATFL04		; no - keep going
			LEAS	1,S				; remove skip factor from stack
			
FORMATFL08	JSR		ENAFLOP			; + enable floppy drive
			CLR		DCOPC			; restore head to track zero DSKCON op code
			CLR		DCTRK			; set DSKCON track variable to track zero
			JSR		[DSKCON]		; restore head to track zero
			TST		DCSTA			; + check status
			LBNE	FORMATFLERR		; + branch on error
			CLR		RDYTMR			; reset the ready timer
			LDA		#$C0			; * FOC read address code
			STA		FDCREG			; *
			JSR		LD7D1			; check drive ready - wait until ready
			BEQ		FORMATFL11		; branch if drives ready
			JMP		FORMATFLERR		; error if drives not ready
			
FORMATFL09	CMPA	#22				; = check for track 22 (precompensation)
			BLO		FORMATFL10		; = and branch if < track 22 - no precomp
			LDA		DRGRAM			; * get the RAM image of DSKREG, 'OR'
			ORA		#$10			; * in the precompensation flag and
			STA		DSKREG			; * send it to DSKREG
FORMATFL10	LDA		#$53			; = get step in command
			STA		FDCREG			; = and send it to the 1793
			EXG		A,A				; * delay after issuing command to 1793
			EXG		A,A				; *
			JSR		LD7D1			; check drive ready
			LBNE	FORMATFLERR		; branch if not ready - issue an error
			
FORMATFL11	LDU		UREG			; restore U register
			JSR		FMTTRKINFO		; display track information
			JSR		[POLCAT]		; read keyboard			
			CMPA	#KEYESC			; check for BREAK key
			BNE		FORMATFL12		; no - skip
			LDA		#120			; 120*1/60 = 2 seconds
			STA		RDYTMR			; wait 2 seconds before turning off motor
			JSR		DISFLOP			; disable floppy drive
			LBRA	FORMATFL17		; exit
			
FORMATFL12	JSR		FDCDELAY		; wait a while
			JSR		BUILDTRK		; build a formatted track in RAM
			LDY		#FDCREG+3		; Y points to 1793 data register
			ORCC	#$50			; disable interrupts
			LDX		#FORMATFL14		; * get return address and store
			STX		DNMIVC			; * it in the non maskable interrupt vector
			LDX		#DISKARR		; point X to the formatted track RAM image
			LDA		FDCREG			; reset status of the 1793
			LDA		#$FF			; * enable the NMI flag to vector
			STA		NMIFLG			; * out of an I/O loop upon an NMI interrupt
			LDB		#$F4			; = get write track command and
			STB		FDCREG			; = send to 1793
			LDA		DRGRAM			; * get the DSKREG RAM image and 'or' in the
			ORA		#$80			; * flag which will enable the 1793 to halt
			STA		DSKREG			; * the 6809. send result to DSKREG
FORMATFL13	LDB		,X+				; = get a byte from the formatted track
			STB		,Y				; = RAM image, send it to the 1793 and
			BRA		FORMATFL13		; = loop back to get another byte
FORMATFL14	LDA		FDCREG			; get status
			ANDCC	#$AF			; enable interrupts
			ANDA	#$44			; * keep only write protect & lost data
			STA		DCSTA			; * and save it in the dskcon status byte
			BNE		FORMATFLERR		; branch if error
			INC		DCTRK			; skip to the next track
			LDA		DCTRK			; get the track number
			CMPA	FLPYTRK			; was it the last track
			BNE		FORMATFL09		; no - keep going			

			LDU		UREG			; restore U register
			LDA		#120			; 120*1/60 = 2 seconds
			STA		RDYTMR			; wait 2 seconds before turning off motor
			
			JSR		DISFLOP			; disable floppy drive

			LDA		FLPYSIDE		; load side selection
			CMPA	#2				; double sided disk?
			BNE		FORMATFL15		; no - skip
			LDA		SIDE			; load side number
			INCA					; increment
			CMPA	#1				; last side?
			BHI		FORMATFL15		; yes - exit
			STA		SIDE			; store value
			LBRA	FORMATFL08		; format next side			
			
FORMATFL15	LDU		UREG			; restore U register
			TST		FLPYVER			; verify floppy?
			BEQ		FORMATFL15B		; no - skip
			JSR		VERIFYFLPY		; verify floppy disk
			BCS		FORMATFL15C		; branch on error
			
FORMATFL15B	LDU		UREG			; restore U register
			LDX		#OPCOMPLBL		; display operation completed
			LDD		#$0409
			JSR		PRINTASC

FORMATFL15C	JSR		CLRSTATUS		; clear status bar

			LDX		#ANYKEYCMD		; display press any key message
			LDD		#$010F
			JSR		PRINTRASC

			JSR		FLUSHSCR		; update screen

			JSR		BEEP			; play sound			

FORMATFL16	JSR		[POLCAT]		; wait for a key
			BEQ		FORMATFL16
			
FORMATFL17	JSR		LISTSDC			; refresh SDC directory
			
FORMATFLEND	RTS
			
FORMATFLERR	LDU		UREG			; restore U register
			LDA		#120			; 120*1/60 = 2 seconds
			STA		RDYTMR			; wait 2 seconds before turning off motor			
			JSR		DISFLOP			; disable floppy drive
			JSR		SHOWERRMSG		; display error message
			BRA		FORMATFL15C

**
*  VERIFY FLOPPY DISK
**
VERIFYFLPY	LDD		#$0409			; load X/Y coordinates
			LDX		#' '+64			; load space - add 64 for non reverse
			LDY		#20				; 20 characters
			JSR		PRINTSTR		; clear line

			LDX		#VERDSKLBL		; load verifying disk label
			LDD		#$0409
			JSR		PRINTASC	

			JSR		FLUSHSCR		; refresh screen

			CLR		SIDE			; start at side 0
			CLR		ERRCTR			; reset error count
			
VERIFYFL01	JSR		ENAFLOP			; enable floppy drive
			
									; * verify that all sectors are readable
									
			LDA		#$02			; = get the DSKCON read op code
			STA		DCOPC			; = and save it in the DSKCON variable
			LDX		#SECBUF			; * point the DSKCON buffer pointer
			STX		DCBPT			; * to SECBUF
			LDU		#DBUF1			; point U to the logical sector numbers
			CLRA					; reset the track counter to zero
VERIFYFL02	STA		DCTRK			; set the DSKCON track variable
			CLRB					; reset the sector counter
VERIFYFL03	LDA		B,U				; get the physical sector number
			STA		DCSEC			; save DSKCON sector variable
			STA 	SECT
			JSR		[DSKCON]		; read a sector
			TST		DCSTA			; check status
			BNE		VERIFYERR		; branch on error
			
			INCB					; * increment the sector counter
			CMPB	#FLPYSECT		; * and compare it to maximum sector number
			BLO		VERIFYFL03		; * and keep looping if more sectors left
			LDA		DCTRK			; = get the current track number
			INCA					; = add one to it, compare to the maximum track
			CMPA	FLPYTRK			; = number and keep looping if
			BLO		VERIFYFL02		; = there are still tracks to do
			
			JSR		DISFLOP			; disable floppy drive
			
			LDA		FLPYSIDE		; load side selection
			CMPA	#2				; double sided disk?
			BNE		VERIFYFL04		; no - skip
			LDA		SIDE			; load side number
			INCA					; increment
			CMPA	#1				; last side?
			BHI		VERIFYFL04		; yes - exit
			STA		SIDE			; store value
			BRA		VERIFYFL01		; verify next side						
			
VERIFYFL04	CLRA					; clear carry
			RTS

VERIFYERR	JSR		DISFLOP			; disable floppy drive
			LDU		UREG			; restore U register
			JSR		SHOWERRMSG		; display error message
			ORCC	#1				; set carry
			RTS

**
*  BUILD A FORMATTED TRACK OF DATA IN RAM
*  FROM DISK BASIC 1.1 DISSASEMBLY - $D691
**
BUILDTRK	LDX		#DISKARR		; start track buffer at DISKARR
			LDD		#$204E			; get set to write 32 bytes of $4E
			JSR		WRITEBYT		; go write gap IV
			CLRB					; reset sector counter
			
BUILDTRK02	PSHS	B				; save sector counter
			LDU		#DBUF1			; point U to the table of logical sectors
			LDB		B,U				; * get logical sector number from table and
			STB		DCSEC			; * save it in the DSKCON variable
			LDU		#LD6D4			; point U to table of sector formatting data
			LDB		#3				; * get first 3 data blocks and
			JSR		WRITEBLK		; * write them to buffer
			LDA		DCTRK			; = get track number and store lt
			STA		,X+				; = in the ram buffer
			LDA		FLPYSIDE		; load side selection
			CMPA	#2				; double sided disk?
			BEQ		BUILDTRK03		; yes - branch
			CLRA					; set side 0
			BRA		BUILDTRK04

BUILDTRK03	LDA		SIDE			; load side number

BUILDTRK04	STA		,X+				; store it in the buffer
			LDA		DCSEC			; * get sector number and
			STA		,X+				; * store it in the buffer
			LDB		#9				; = get the last nine data blocks and
			JSR		WRITEBLK		; = write them to the buffer
			PULS	B				; get sector counter
			INCB					; next sector
			CMPB	#FLPYSECT		; 18 sectors per track
			BLO		BUILDTRK02 		; branch if all sectors not done
			LDD		#$C84E			; write 200 bytes of $4E at end of track
			JSR		WRITEBYT		; go do it
			RTS
			
**
*  WRITE BYTES INTO BUFFER
*    A = NUMBER OF BYTES
*    B = BYTE TO WRITE
*    X = BUFFER
**
WRITEBYT 	STB		,X+				; store a byte in the buffer
			DECA					; decrement counter
			BNE		WRITEBYT		; branch if all bytes not moved
			RTS

**
*  WRITE DATA BLOCKS INTO BUFFER
*    B = NUMBER OF BLOCKS TO WRITE
*    X = BUFFER
*    U = TABLE OF SECTOR FORMATTING DATA
**
WRITEBLK	PSHS	B				; save the counter on the stack
			LDD		,U++			; get two bytes of data from the table
			BSR		WRITEBYT		; write ACCA bytes of ACCB into the buffer
			PULS	B				; * get the counter back, decrement
			DECB					; * it and branch if all data blocks
			BNE		WRITEBLK		; * not done
			RTS

**
*  DDCDELAY
**
FDCDELAY	LDX		#8750			; delay for a while
FDCDELAY02	LEAX	-1,X			; * decrement delay counter and
			BNE		FDCDELAY02		; * branch if not done			
			RTS
			
**
*  DISPLAY FORMATTING TRACK INFO
**
FMTTRKINFO	CLRA					; display track number
			LDB		DCTRK
			LDX		#STR1
			JSR		ITOA8

			LEAX	1,X
			LDD		#$1509
			JSR		PRINTASC

			JSR		FLUSHSCR		; refresh screen

			RTS

**
*  SHOW ERROR MESSAGE
**
SHOWERRMSG	LDD		#$0409			; load X/Y coordinates
			LDX		#' '+64			; load space - add 64 for non reverse
			LDY		#20				; 20 characters
			JSR		PRINTSTR		; clear line
			LDB		DCSTA			; get status
			BMI		SHOWERR04		; branch if drive not ready
			BITB	#$40			; write protect
			BNE		SHOWERR03		; branch if disk is write protected
			LDB		DCOPC			; get disk operation
			CMPB	#3				; write sector?
			BEQ		SHOWERR02		; yes - branch
			
			LDX		#READDSKERR		; load error reading disk message
			BRA		SHOWERR05

SHOWERR02	LDX		#WRITDSKERR		; load error writing disk message
			BRA		SHOWERR05

SHOWERR03	LDX		#WRITPROTERR	; load write protect error message
			BRA		SHOWERR05

SHOWERR04	LDX		#NOTREADYERR	; load drive not ready error message
SHOWERR05	LDD		#$0409			; load X/Y coordinates
			JSR		PRINTASC		; print message
			RTS

**
*  READ A DISK IMAGE SECTOR
*    A = TRACK NUMBER
*    B = SECTOR NUMBER
*  EXIT
*    B = STATUS CODE
*    X = SECTOR BUFFER
**
READLSN		JSR		GETLSN			; get LSN
			TFR		D,Y				; set low-order word of LSN
			CLRB					; set high-order byte of LSN
			LDX		#SECBUF			; load sector buffer
			LDA		#CMDREAD		; read sector command
			JSR		SENDCMD			; send command to Coco SDC
			RTS
			
**
*  WRITE A DISK IMAGE SECTOR
*    A = TRACK NUMBER
*    B = SECTOR NUMBER
*  EXIT
*    B = STATUS CODE
*    X = SECTOR BUFFER
**
WRITELSN	JSR		GETLSN			; get LSN
			TFR		D,Y				; set low-order word of LSN
			CLRB					; set high-order byte of LSN
			LDX		#SECBUF			; load sector buffer
			LDA		#CMDWRITE		; write sector command
			JSR		SENDCMD			; send command to Coco SDC
			RTS

***
*  GET LSN NUMBER FOR SPECIFIED TRACK/SECTOR
*    A = TRACK NUMBER
*    B = SECTOR NUMBER
*  EXIT
*    D = LSN
***
GETLSN		DECB					; decrement sector number
			PSHU	B				; save sector number
			LDB		FLPYSIDE		; get side selection
			CMPB	#2				; double sided disk?
			BNE		GETLSN01		; no - skip
			LSLA					; multiply track number by 2
			ADDA	SIDE			; add side number
			
GETLSN01	LDB		#18				; 18 sectors/track
			MUL						; TRACK * 18
			ADDB	,U+				; add SECTOR and restore stack
			ADCA	#0				; add carry

			RTS

**
*  ENABLE FLOPPY DRIVE
**
ENAFLOP		PSHU	A,B,CC			; save registers

			LDA		MPIREG			; save current MPI slot
			STA		MPISLOT

			LDX		#SDCDRV0		; load SDC drive access table
			LDB		FDNUM			; get drive number
			LDA		B,X				; get value for drive FDNUM
			ORA		#$80			; set bit 7			
			STA		B,X				; update SDC drive access table
			STB		DCDRV			; set DSKCON drive number

			TST		SIDE			; second side of the disk?
			BEQ		ENAFLOP02		; no - exit

			LDX		#DRVTBL			; load drive access table
			LDA		B,X				; get drive FDNUM pattern
			ORA		#$40			; set side select bit
			STA		B,X				; update drive access table

ENAFLOP02	PULU	A,B,CC			; restore registers
			RTS
			
**
*  DISABLE FLOPPY DRIVE
**
DISFLOP		PSHU	A,B,CC			; save registers

			LDX		#SDCDRV0		; load SDC drive access table
			LDB		FDNUM			; get drive number
			LDA		B,X				; get value for drive FDNUM
			ANDA	#~$80			; clear bit 7
			STA		B,X				; update SDC drive access table

			LDA		WORKDRV			; load work drive
			STA		DCDRV			; set DSKCON drive number			

			TST		SIDE			; second side of the disk?
			BEQ		DISFLOP02		; no - exit

			LDX		#DRVTBL			; load drive access table
			LDA		B,X				; get drive FDNUM pattern
			ANDA	#~$40			; clear side select bit
			STA		B,X				; update drive access table
			
DISFLOP02	LDA		MPISLOT			; restore MPI slot
			STA		MPIREG

			PULU	A,B,CC			; restore registers
			RTS

**
*  ENSURE ALL RAM MODE FOR DOUBLE SIDED DISKS OPERATIONS
**
CHKALLRAM	TST		FLPYSIDE		; double sided disk?
			BNE		CHKALLRAM01		; yes - set ALL RAM mode 
			TST		SIDE			; second side of the disk?
			BEQ		CHKALLEND		; no - exit			
			
CHKALLRAM01	TST		ALLRAMFLAG		; already in ALL RAM mode?
			BNE		CHKALLEND		; yes - exit
			LDD		$FFFE			; check if Coco 3
			CMPD	#$8C1B			; Coco 3 reset vector
			BEQ		CHKALLRAM02		; yes - set ALL RAM flag
			JSR		ALLRAM			; enable ALL RAM mode

CHKALLRAM02	INC		ALLRAMFLAG		; set ALL RAM flag

CHKALLEND	RTS

**
*  ENABLE ALL RAM MODE
**
ALLRAM		ORCC	#INTMASKS		; mask irqs
			LDX		#$8000			; load start of ROM

ALLRAM02	STA		ROMSET			; enable ROM
			LDD		,X				; read 2 bytes from ROM
			STA		ROMCLR			; disable ROM
			STD		,X++			; write 2 bytes to RAM
			CMPX	#$E000			; check for ROM end
			BNE		ALLRAM02		; repeat until done

			ANDCC	#~INTMASKS		; unmask irqs
			RTS
		
**
*  PLAY A BEEP SOUND
**			
BEEP		PSHU	X,Y,A,B,CC		; save registers

			LDD		#25				; play first note
			STD		SNDREPEAT
			LDA		#8
			STA		SNDDELAY
			JSR		SOUND
			
			LDD		#3000			; silence delay
			JSR		DELAY
			
			LDD		#25				; play second note
			STD		SNDREPEAT
			LDA		#4
			STA		SNDDELAY
			JSR		SOUND
			
			PULU	X,Y,A,B,CC		; restore registers
			
			RTS
			
**
*  PLAY SOUND ON THE TV SPEAKER
**			
SOUND		LDA		$FF01			; select sound out
			ANDA	#$F7			; reset mux bit
			STA		$FF01			; store
			LDA		$FF03			; select sound out
			ANDA	#$F7			; reset mux bit
			STA		$FF03			; store
			LDA		$FF23			; get PIA
			ORA		#8				; set 6-bit sound enable
			STA		$FF23			; store
			
			LDY		SNDREPEAT		; get # of time
			
SOUND02		LDX		SNDSTART		; get start address

SOUND03		LDA		,X+				; get next byte
			ANDA	#$FC			; reset 2 ls bits
			STA		$FF20			; output
			LDA		SNDDELAY		; get delay count

SOUND04		DECA					; decrement counter
			BNE		SOUND04			; go if not 0

			CMPX	SNDEND			; test for end
			BNE		SOUND03			; go if not end
			LEAY	-1,Y			; decrement # of times
			BNE		SOUND02			; go if not 0
			
			LDA		$FF23			; get pia
			ANDA	#~8				; clear 6-bit sound enable
			STA		$FF23			; store
			
			RTS

**
*  DELAY
*   D=DELAY COUNTER
**			
DELAY		SUBD	#1				; decrement counter
			BNE		DELAY			; continue until done
			RTS			
			endif

**
*  PRINT COMMAND SHORTCUT
**
PRINTCMD	PSHU	X				; save command string
			LDX		#SHIFTLBL		; load SHIFT label
			JSR		PRINTASC		; print it
			PULU	X				; restore command string
			ADDA	#6				; move 6 characters right
			JSR		PRINTASC		; print command
			RTS
			
**
*  SHOW HELP
**
HELP		JSR		DRAWBG

			LDD		#$0102
			LDX		#31
			LDY		#12
			JSR		WINDOW

			LDX		#HELPLBL
			LDD		#$0102
			JSR		PRINTRASC

			LDD		#$0203
			LDX		#HELPMSG1
			JSR		PRINTCMD

			LDD		#$0204
			LDX		#HELPMSG2
			JSR		PRINTCMD

			LDD		#$0205
			LDX		#HELPMSG3
			JSR		PRINTCMD

			LDD		#$0206
			LDX		#HELPMSG4
			JSR		PRINTCMD

			LDD		#$0207
			LDX		#HELPMSG5
			JSR		PRINTCMD

			LDD		#$0208
			LDX		#HELPMSG6
			JSR		PRINTCMD

			LDD		#$0209
			LDX		#HELPMSG7
			JSR		PRINTCMD

			LDD		#$0A0A
			LDX		#HELPMSG8
			JSR		PRINTRASC

			LDD		#$020B
			LDX		#HELPMSG9
			JSR		PRINTCMD

			LDD		#$020C
			LDX		#HELPMSG10
			JSR		PRINTASC

			JSR		CLRSTATUS		; clear status bar
			
			LDX		#ABOUTCMD		; display about command
			LDD		#$010F
			JSR		PRINTRASC

			JSR		FLUSHSCR

HELPKEY		JSR		[POLCAT]
			BEQ		HELPKEY
			CMPA	#KEYESC
			LBEQ	HELPEND
			CMPA	#'A'
			BNE		HELP02
			JSR		ABOUT
			LBRA	HELPEND

HELP02 		JSR		DRAWBG

			if		FLOPPYDRV
			LDD		#$0102
			LDX		#31
			LDY		#6
			JSR		WINDOW
			
			LDX		#HELPFLLBL
			LDD		#$0102
			JSR		PRINTRASC			
			
			LDD		#$0203
			LDX		#HELPFLMSG1
			JSR		PRINTCMD

			LDD		#$0204
			LDX		#HELPFLMSG2
			JSR		PRINTCMD

			LDD		#$0205
			LDX		#HELPFLMSG3
			JSR		PRINTCMD
			
			LDD		#$0206
			LDX		#HELPFLMSG4
			JSR		PRINTCMD
			endif

			LDD		#$0105+(FLOPPYDRV*3)	; navigation keys window
			LDX		#31
			LDY		#7
			JSR		WINDOW
			
			LDX		#NAVLBL
			LDD		#$0105+(FLOPPYDRV*3)
			JSR		PRINTRASC

			LDD		#$0206+(FLOPPYDRV*3)
			LDX		#NAVMSG1
			JSR		PRINTCMD

			LDD		#$0207+(FLOPPYDRV*3)
			LDX		#NAVMSG2
			JSR		PRINTCMD

			LDD		#$0208+(FLOPPYDRV*3)
			LDX		#NAVMSG3
			JSR		PRINTCMD

			LDD		#$0209+(FLOPPYDRV*3)
			LDX		#NAVMSG4
			JSR		PRINTCMD

			LDD		#$020A+(FLOPPYDRV*3)
			LDX		#NAVMSG5
			JSR		PRINTASC
			
			JSR		CLRSTATUS		; clear status bar
			
			LDX		#ABOUTCMD		; display about command
			LDD		#$010F
			JSR		PRINTRASC			
			
			JSR		FLUSHSCR

HELPKEY2	JSR		[POLCAT]
			BEQ		HELPKEY2
			CMPA	#KEYESC
			BEQ		HELPEND
			CMPA	#'A'
			BNE		HELP03
			JSR		ABOUT
			BRA		HELPEND
			
HELP03		JSR		DRAWBG

			LDD		#$0202			; joystick support window
			LDX		#29
			LDY		#12
			JSR		WINDOW
			
			LDD		#$0202
			LDX		#JOYLBL
			JSR		PRINTRASC
			
			LDD		#$0304
			LDX		#JOYMSG1
			JSR		PRINTASC

			LDD		#$0305
			LDX		#JOYMSG2
			JSR		PRINTASC
			
			LDD		#$0307
			LDX		#JOYMSG3
			JSR		PRINTASC

			LDD		#$0308
			LDX		#JOYMSG4
			JSR		PRINTASC

			LDD		#$0309
			LDX		#JOYMSG5
			JSR		PRINTASC

			LDD		#$030B
			LDX		#JOYMSG6
			JSR		PRINTASC

			JSR		CLRSTATUS		; clear status bar

			LDX		#ABOUTCMD		; display about command
			LDD		#$010F
			JSR		PRINTRASC

			JSR		FLUSHSCR

HELPKEY3	JSR		[POLCAT]
			BEQ		HELPKEY3
			CMPA	#'A'
			BNE		HELPEND
			JSR		ABOUT

HELPEND		RTS

**
*  SHOW ABOUT DIALOG
**
ABOUT		JSR		DRAWBG

			LDD		#$0304
			LDX		#26
			LDY		#9
			JSR		WINDOW

			LDX		#' '				; clear window title
			LDY		#25
			LDD		#$0304
			JSR		PRINTSTR
			
			LDX		#ABOUTLBL			; display window title
			LDD		#$0D04
			JSR		PRINTRASC

			LDD		#$0506
			LDX		#ABOUTMSG1
			JSR		PRINTASC
			LDD		#$0507
			LDX		#ABOUTMSG2
			JSR		PRINTASC

			LDD		#$0509
			LDX		#ABOUTMSG3
			JSR		PRINTASC
			LDD		#$050A
			LDX		#ABOUTMSG4
			JSR		PRINTASC

			JSR		FLUSHSCR

ABOUTKEY	JSR		[POLCAT]
			BEQ		ABOUTKEY
			RTS

			if		DEBUG
PRINTDBG	PSHU	A
			LDA		DEBUGINFO
			STA		VIDRAM+31
			STA		SCRBUF+31
			PULU	A
			RTS
			endif

**
*  STRINGS AND VARIABLES
**
			if		COCO
RUNCMD		FCN		/CLS:DRIVE 0:RUN"/
RUNCMD2		FCN		/"/
RUNMCMD		FCN		/CLS:DRIVE 0:LOADM"/
RUNMCMD2	FCN		/":EXEC/
DOSCMD		FCN		"CLS:DRIVE 0:DOS"
			else
RUNCMD		FCN		/CLS:RUN"/
RUNCMD2		FCN		/"/
DOSCMD		FCN		"CLS:BOOT"
FLPYENACMD	FCN		"SDRIVE 1,OFF"
FLPYDISCMD	FCN		"SDRIVE 1,ON"
			endif
CLSCMD		FCN		"CLS"

; SDC COMMANDS
LISTCMD		FCN		"L:*.*"
CREATECMD	FCN		"N:"
MAKEDIRCMD	FCN		"K:"

			if		COCO
DSKEXT		FCN		".DSK"
			else
DSKEXT		FCN		".VDK"
			endif

INVCHARS	FCC		"*+,./:;<=>?[\]"
			FCB		#'"',0

PRGTITLE	FCC		" SDCX "
			FCB		VERSION,'.',MAJOR,MINOR
			FCN		" - COCO SDC EXPLORER  "
DRVLABEL	FCN		"   SD CARD    "
DIRLABEL	FCN		"  DIRECTORY   "
HELPCMD		FCN		"SHIFT-H=HELP"
EXITCMD		FCN		"BREAK=EXIT"
BOOTCMD		FCN		"ENTER=BOOT"
OS9DSKMSG	FCN		"OS-9 DISK"
OS9BOOTMSG	FCN		"PRESS ENTER"
OS9BOOTMSG2 FCN		"TO BOOT"
MNTERRMSG	FCN		"INVALID DISK"
MNTDRV1MSG	FCN		"DISK MOUNTED"
MNTDRV1MSG2 FCN		"IN DRIVE ",LBLDRV1
EMPTYCARD	FCN		" EMPTY CARD"
JOYMSG		FCN		"JOYSTICK ENABLED"

CREATELBL	FCN		"  CREATE DISK  "
MAKEDIRLBL	FCN		"   MAKE DIR.   "
RENDSKLBL	FCN		"  RENAME DISK  "
RENDIRLBL	FCN		"  RENAME DIR.  "
FILEINFOLBL	FCN		"     FILE INFO      "
NAMELBL		FCN		"NAME:"
TYPELBL		FCN		"TYPE:"
SIZELBL		FCN		"SIZE:"
FILELBL		FCN		"FILE"
DIRLBL		FCN		"DIRECTORY"
NALBL		FCN		"N/A"
KBLBL		FCN		"KB"
MBLBL		FCN		"MB"
SORTLBL		FCN		"SORTING DIRECTORY..."
SORTEDLBL	FCN		"^"
CONFIRMLBL	FCN		"     CONFIRMATION     "
DELDSKMSG	FCN		"DELETE DISK (Y/N)?"
DELDIRMSG	FCN		"DELETE DIR. (Y/N)?"

			if		FLOPPYDRV
WRITFLPYLBL	FCN		"    WRITE FLOPPY DISK     "
READFLPYLBL	FCN		"     READ FLOPPY DISK     "
FMTFLPYLBL	FCN		"    FORMAT FLOPPY DISK    "
FLPYDIRLBL	FCN		"       FLOPPY DRIVE 0       "
FLPYREADLBL	FCN		"READING DIRECTORY..."
FILENAMELBL	FCN     "FILENAME:"
TRACKSLBL	FCN		"tRACKS:"
SIDELBL		FCN		"sIDE(S):"
SSLBL		FCN		"SINGLE"
S2LBL		FCN		"SIDE 2"
DSLBL		FCN		"DOUBLE"
VERIFYLBL	FCN		"vERIFY:"
YESLBL		FCN		"YES"
NOLBL		FCN		"NO "
FLPYMSG1	FCN		"INSERT DISK IN DRIVE"
FLPYMSG2	FCN		"PRESS [ENTER] WHEN READY"
ABORTCMD	FCN		"BREAK=ABORT"
SIDESELCMD	FCN		"S=SIDE:"
DRVSELCMD	FCN		"0-","0"+FLPYMAX,"=DRIVE"
UPDOWNCMD	FCN		"UP/DOWN"
FLPYINFOLBL	FCN		"TRACK 00 - SECTOR 01"
FMTINFOLBL	FCN		"FORMATTING TRACK 00"
VERDSKLBL	FCN		"VERIFYING..."
OPCOMPLBL	FCN		"OPERATION COMPLETED."
ANYKEYCMD	FCN		"PRESS ANY KEY"

READERRLBL	FCN		"READ ERRORS:"
WRITERRLBL	FCN		"WRITE ERRORS:"
ERRORLBL	FCN		" ERROR "
READDSKERR	FCN		"ERROR READING DISK "
WRITDSKERR	FCN		"ERROR WRITING DISK "
NOTREADYERR	FCN		"DRIVE NOT READY"
WRITPROTERR	FCN		"WRITE PROTECT ERROR"
			endif

SDCERRLBL	FCN		"        ERROR         "
SDCBUSYERR	FCN		"  SDC NOT READY"
DIRNFERR	FCN		"DIRECTORY NOT FOUND"
DIRNEMPTERR	FCN		"DIRECTORY NOT EMPTY"
INVPATHERR	FCN		"PATHNAME IS INVALID"
HWERR		FCN		"  HARDWARE ERROR"
ALROPENERR	FCN		" FILE ALREADY OPEN"
ALREXISTERR	FCN		"FILE ALREADY EXISTS"
UNKNOWNERR	FCN		"  UNKNOWN ERROR"
NOSDCERR	FCN		" COCO SDC NOT FOUND"
SDCDOSREQ	FCN		"  SDC-DOS REQUIRED"
MPIREQ		FCN		" MULTI-PAK REQUIRED"
READERRMSG	FCN		"ERROR READING SECTOR"
WRITERRMSG	FCN		"ERROR WRITING SECTOR"
MAXFLERRMSG	FCN		" MAXIMUM 512 FILES!"

; HELP

HELPLBL		FCN		"         DISK COMMANDS        "
HELPMSG1	FCN		"C: CREATE DISK"
HELPMSG2	FCN		"K: MAKE DIRECTORY"
HELPMSG3	FCN		"N: RENAME DISK OR DIR."
HELPMSG4	FCN		"X: DELETE DISK OR DIR."
HELPMSG5	FCN		LBLDRV1,": MOUNT DISK IN DRV ",LBLDRV1
HELPMSG6	FCN		"I: FILE INFORMATION"
HELPMSG7	FCN		"S: TOGGLE SORTING"
HELPMSG8	FCN		" MULTI DISKS "
HELPMSG9	FCN		"M: MOUNT DIRECTORY"
HELPMSG10	FCN		"CLEAR:   REFRESH DIRECTORY"

NAVLBL		FCN		"        NAVIGATION KEYS       "
NAVMSG1		FCN		"UP:    PAGE UP"
NAVMSG2		FCN		"DOWN:  PAGE DOWN"
NAVMSG3		FCN		"LEFT:  HOME"
NAVMSG4		FCN		"RIGHT: END"
NAVMSG5		FCN		"[A-Z][0-9]:  QUICK SELECTION"

			if		FLOPPYDRV
HELPFLLBL	FCN		"    FLOPPY DRIVE COMMANDS     "
HELPFLMSG1	FCN		"D: FLOPPY DISK DIR."
HELPFLMSG2	FCN		"R: READ FLOPPY DISK"
HELPFLMSG3	FCN		"W: WRITE FLOPPY DISK"
HELPFLMSG4	FCN		"F: FORMAT FLOPPY DISK"
			endif

SHIFTLBL	FCN		"SHIFT-"

; JOYSTICK

JOYLBL		FCN		"      JOYSTICK SUPPORT      "
JOYMSG1		FCN		"PRESS RIGHT OR LEFT BUTTON"
JOYMSG2		FCN		"TO ENABLE JOYSTICK."
JOYMSG3		FCN		"PRESS BUTTON TO LAUNCH"
JOYMSG4		FCN		"PROGRAM, CHANGE DIRECTORY"
JOYMSG5		FCN		"OR BOOT DISK."
JOYMSG6		FCN		"HOLD BUTTON FOR FAST NAV."

; ABOUT

ABOUTCMD	FCN		"A=ABOUT"
ABOUTLBL	FCN		"ABOUT"
ABOUTMSG1	FCN		"COCO SDC EXPLORER"
ABOUTMSG2	FCN		"2016 GUILLAUME MAJOR"
ABOUTMSG3	FCN		"BASED ON SIDEKICK BY"
ABOUTMSG4	FCN		"LUIS ANTONIOSI"

CURLINE		FDB		0
STACKPTR	FDB		0

DSKSIZE		FDB		(DISKARREND-DISKARR)/16
DSKSEL		FDB		0
DSKSELPREV	FDB		0
DSKOFF		FDB		0
DSKADR		FDB		0

DIRSIZE		FCB		0
DIRSEL		FCB		0
DIRSELPREV	FCB		0
DIROFF		FCB		0
WORKDRV		FCB		DRAGON			; Working Drive Number - Coco=0, Dragon=1
SDCDRV		FCB		0				; Coco SDC Drive Number

			if		FLOPPYDRV
FDNUM		FCB		0				; Floppy drive number
FLOPPY		FCB		0
FLPYTRK		FCB		35				; number of tracks per disk (floppy)
FLPYSIDE	FCB		0				; side selection (floppy)
FLPYVER		FCB		0				; verify floppy flag
FORMATWIN	FCB		0				; format window flag
ALLRAMFLAG	FCB		0				; all RAM flag
MPI			FCB		0				; MPI present flag
UREG		FDB		0				; U register backup
			endif

KEYPRES		FDB		0
SHIFTKEY	FCB		0
JOYBUT		FCB		0
JOYSEL		FCB		0
JOYFLAG		FCB		0
JOYTMR		FCB		0				; Joystick timer
JOYBTMR		FCB		0				; Joystick button timer
JOYSTA		FCB		0
PANEFOCUS	FCB		0
CURSEC		FCB		0
SECPOS		FCB		0

OS9DSK		FCB		0
BOOTABLE	FCB		0
MULTIDSK	FCB		0
MNTERR		FCB		0
MNTDRV1		FCB		0
DSKERR		FCB		0
FATAL		FCB		0
MAXFILERR	FCB		0
RETRYCNT	FCB		0
DIRSORT		FCB		0
MPISLOT		FCB		0

PARENT		FDB		0
CHILD		FDB		0
LAST		FDB		0
CTR			FDB		0
DIRBIT		FCB		0

SREG		FDB		0

FILENAME	FCB		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
CURSTR		FCN		" "
INPUTPOS	FDB		0
KBUF		FCB		0,0,0,0,0,0
KBUFPOS		FCB		0
KBUFTMR		FDB		0
KBUFDELAY	FDB		0
KEYSRCH		FDB		0
SRCHPASS	FCB		0

			if		FLOPPYDRV
TRACK		FCB		0	
SECT		FCB		0
SIDE		FCB		0
ERRCTR		FCB		0

SNDREPEAT	FDB		0
SNDDELAY	FCB		0
SNDSTART	FDB		SINETBL
SNDEND		FDB		SINETBL+36

* SINE TABLE FOR SOUND
					
SINETBL		FCB		$82,$92,$AA,$BA,$CA,$DA
			FCB		$EA,$F2,$FA,$FA,$FA,$F2
			FCB		$EA,$DA,$CA,$BA,$AA,$92
			FCB		$7A,$6A,$52,$42,$32,$22
			FCB		$12,$0A,$02,$02,$02,$0A
			FCB		$12,$22,$32,$42,$52,$6A
			endif

			if		!TEST
PDISKARR	FDB		0				; disk array pointer
DISKARR		RMB		SDCDIRMAX*16	; disk array
			else
PDISKARR	FDB		#DISKARR
			include "diskarray.asm"
			endif
DISKARREND

STR1		RMB		256				; temp string buffer
			if		COCO
SECBUF		EQU		$0600			; use Disk Basic disk buffer
			else
SECBUF		RMB		256				; disk sector buffer
			endif
SCRBUF		RMB		32*16			; screen buffer
DIRARR		RMB		160*16			; file directory array

DSKFN		RMB		11				; current disk file name

STACK		RMB		64
USRSTK		EQU 	*

			END START
